﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "OSDependencies.h"


/**
    @file
    @brief header defining nn::profiling::DumpThread class,
           which has the ability to start/stop/activate a background thread
           for the purpose  of dumping the log.
           The background thread is created along with an event handle,
           which can be used to trigger the thread
           - even from a different process.
*/


namespace nn
{

namespace perflog
{


template<typename _DumpCallback = void(*)(void*), typename _DumpFilter = void *> class DumpThread
{
private:
    enum EState
    {
        EStateOff       = 0,
        EStateStarting  = 1,
        EStateOn        = 2,
        EStateStopping  = 3,
        EStateStopped   = 4,
        EStateResetting = 5,
        EStateInvalid   = 6
    };
    #if defined(USE_ATOMIC)
        typedef std::atomic<EState>  AtomicStateType;
    #else
        typedef EState               AtomicStateType;
    #endif
    static const uint32_t WaitTimeoutMs = (30 * 1000);

public:
    explicit DumpThread(_DumpCallback callback, _DumpFilter filter) NN_NOEXCEPT : // constructor
        m_pCallback        ( callback ),
        m_Filter           ( filter )
    {
        ResetState();
    }


    virtual ~DumpThread() NN_NOEXCEPT
    {
        Stop();
    }

    bool Start(OSEvent *pNewWakeUpEvent) NN_NOEXCEPT
    {
        EState curr = StateSwitch(EStateOff, EStateStarting);
        if( curr == EStateStopped )
        {
            if( StateSwitch(EStateStopped, EStateResetting) == EStateStopped )
            {
                // close all handles, try again
                ResetState();
                curr = StateSwitch(EStateOff, EStateStarting);
            }
            else
            {
                return false;
            }
        }
        if( curr != EStateOff )
        {
            return false;
        }
        if( !OSCreateEvent(&m_DumpEvent) )
        {
            ResetState();
            return false;
        }
        if( !OSCreateThread( &m_DumpThread, DumpThreadStarter, this ) )
        {
            ResetState();
            return false;
        }

        // user will signal this to dump
        (*pNewWakeUpEvent) = m_DumpEvent;
        return true;
    } // NOLINT[impl/function_size]

    bool Stop() NN_NOEXCEPT
    {
        EState curr = StateSwitch(EStateOn, EStateStopping);
        if( curr != EStateStopped )
        {
            bool bTimedOut = false;
            if( curr != EStateOn )
            {
                // we have not touched the state!
                return false;
            }

            // fail here is fine, the thread will join upon timeout if it is running
            OSSignalEvent( &m_DumpEvent );

            if( !OSJoinThread( &m_DumpThread, WaitTimeoutMs * 2, &bTimedOut ) )
            {
                // can't join now.
                // If thread exits the future, user can call Stop() or Start() again.
                return false;
            }
            OSCloseThread(&m_DumpThread);
            OSCloseEvent (&m_DumpEvent);
        }
        if( StateSwitch(EStateStopped, EStateResetting) == EStateStopped )
        {
            ResetState();
        }
        return true;
    }

    bool IsRunning() NN_NOEXCEPT
    {
        EState currState;
        #if defined(USE_ATOMIC)
            currState = m_State.load();
        #else
            currState = m_State;
        #endif
        return ( currState == EStateOn );
    }

private:
    void ResetState() NN_NOEXCEPT
    {
        #if defined(USE_ATOMIC)
            m_State.store( EStateOff );
        #else
            m_State = EStateOff;
        #endif

        // m_DumpEvent  = OSInvalidEvent;
        // m_DumpThread = OSInvalidThread;
    }

    // returns the state prior to this operation. if it matches stateFrom, operation is successful
    EState StateSwitch( EState stateFrom,  EState stateTo ) NN_NOEXCEPT
    {
        #if defined(USE_ATOMIC)
            bool bSuccess;
            EState expected = stateFrom;
            bSuccess = m_State.compare_exchange_strong( expected, stateTo );
            if( (!bSuccess) && (expected == stateFrom) )
            {
                // this should never happen, but doing so ensures correct behavior
                expected = EStateInvalid;
            }
            return expected;
        #else
            return static_cast<EState>(__sync_val_compare_and_swap(&m_State, stateFrom, stateTo));
        #endif
    }

    static uint32_t DumpThreadStarter( void *arg ) NN_NOEXCEPT
    {
        static_cast<DumpThread *>(arg)->BackgroundThread();
        return 0;
    }

    void BackgroundThread() NN_NOEXCEPT
    {
        bool bTimedOut = false;
        if( StateSwitch(EStateStarting, EStateOn) != EStateStarting )
        {
            return;
        }
        while( IsRunning() )
        {
            if( OSWaitForEvent( &m_DumpEvent, WaitTimeoutMs, &bTimedOut) )
            {
                if (IsRunning())
                {
                    m_pCallback(m_Filter);
                }
                else
                {
                    break; // no point checking again
                }
            }
            else if( !bTimedOut )
            {
                // something bad happened, prevent infinite loop
                break;
            }
        }
        // best-effort signal that thread is done
        StateSwitch(EStateStopping, EStateStopped);
    } // NOLINT[impl/function_size]


private:
    _DumpCallback   m_pCallback; // when it's time to dump the log, this is called...
    _DumpFilter     m_Filter;   // ... passing this argument

    OSEvent         m_DumpEvent;
    OSThread        m_DumpThread;
    bool            m_DumpThreadActive;
    AtomicStateType m_State;
}  // NOLINT[impl/function_size]
;




// utility to help user pass class members as callbacks
// (a poor man's version of boost::bind, made for this specific use)
template<typename _C, typename _FilterInfo = typename _C::FilterType > class DumpFnBind
{
public:
    typedef void (_C::*DumpMethod)(_FilterInfo);
public:
    DumpFnBind(_C *p, DumpMethod m) :
        m_pObject(p), m_pMethod(m)
    {
    }
    void operator()(_FilterInfo filter)
    {
        (m_pObject->*m_pMethod)(filter);
    }
private:
    _C        *m_pObject;
    DumpMethod m_pMethod;
private:
    DumpFnBind(); // no default constructor

    // default assignment & copy constructors are fine
};


template<typename _C, typename _FilterInfo>
DumpFnBind<_C, _FilterInfo> MakeFnBind(_C *p, typename DumpFnBind<_C, _FilterInfo>::DumpMethod m)
{
    return DumpFnBind<_C, _FilterInfo>(p, m);
}



} /* namespace perflog */


} /* namespace nn */




