﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   PCV ドライバライブラリの API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include <nn/pcv/pcv_Types.h>

namespace nn {
namespace pcv {
namespace driver {

/**
 * @brief       Enable or disable power on a Module.
 *
 * @param[in]   moduleId The Module to modify.
 * @param[in]   enabled Set to true to enable power for this Module.
 *
 * @post        If the Module supports power and if setting power state
 *              is allowed in the current state, the Module will be
 *              powered on or off based on the enabled flag.
 *
 */
Result SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]
 * @param[out]
 * @param[out]
 * @param[in]
 * @param[in]
 *
 * @post
 */
Result GetPossibleClockRates(ClockRatesListType* pOutType, ClockHz* pOutRates,
                             int* pOutCount, Module moduleId, int maxCount) NN_NOEXCEPT;

/**
 * @brief       Enable or disable clocks on an moduleId.
 *
 * @param[in]   moduleId The Module to modify.
 * @param[in]   enabled Set to true to enable clock for this Module.
 *
 * @post        If the Module supports clocks and clock gating
 *              the clock for this Module will be enabled based on
 *              the enabled flag.
 *
 */
Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;

/**
 * @brief           Set the clock rate on an individual Module.
 *
 * @param[in]       moduleId The Module to modify.
 * @param[in]       clockRateHz the clock rate to use.
 *                  clockRateHz must be one of the possible rates returned
 *                  by QueryPossibleRates for this Module.
 *
 * @post            The clock for this Module will be set to clockRateHz
 *                  if it is one of the QueryPossibleRates for this Module.
 *
 */
Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]  pClockRateHz
 * @param[in]   moduleId
 * @return
 * @retval      ResultSuccess
 *
 * @pre
 * @post
 * @details
 */
Result GetClockRate(ClockHz* pClockRateHz, Module moduleId) NN_NOEXCEPT;

/**
 * @brief           Query the current state of an Module.
 *
 * @param[out]      pState return current state of Module
 * @param[in]       moduleId The Module to query
 *
 * @post            pState holds the clock and power state for an Module.
 *
 */
Result GetState(ModuleState *pState, Module moduleId) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[in]   moduleId
 * @param[in]   clockRateHz
 * @return
 * @retval      ResultSuccess
 *
 * @pre
 * @post
 * @details
 */
Result SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[in]
 * @param[in]
 *
 * @post
 *
 */
Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[in]   powerDomain
 * @param[in]   enabled
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 */
Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]  enabled
 * @param[in]   powerDomain
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 */
Result GetVoltageEnabled(bool* pEnabled, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]  pOutType クロックのリストの表記形式
 * @param[out]  pOutValues クロックのリスト
 * @param[out]  pOutCount クロックのリストの長さ
 * @param[in]   powerDomain
 * @param[in]   maxCount クロックのリストの最大の長さ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 * @details
 */
// Result GetPossibleVoltageValues(VoltageValuesListType* pOutType, MicroVolt* pOutValues,
//                                int* pOutCount, PowerDomain powerDomain, int maxCount) NN_NOEXCEPT;
Result GetVoltageRange(MicroVolt* pMinVolt, MicroVolt* pMaxVolt, MicroVolt* pStepVolt, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[in]   powerDomain
 * @param[in]   microVolt
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 * @details
 */
Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]  pMicroVolt
 * @param[in]   powerDomain
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 * @details
 */
Result GetVoltageValue(MicroVolt* pMicroVolt, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[out]  pThreasholds
 * @param[out]  pOutCount
 * @param[in]   maxCount
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 * @details
 */
Result GetTemperatureThresholds(TemperatureThreshold* pThresholds, int* pOutCount, int maxCount) NN_NOEXCEPT;

/**
 * @brief
 *
 * @param[in]   tempMilliC
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre
 * @post
 * @details
 */
Result SetTemperature(MilliC temperatureMilliC) NN_NOEXCEPT;

} // driver
} // pcv
} // nn
