﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/pctl_TypesWatcher.h>

namespace nn { namespace pctl {

/**
 * @brief 連携先相手のアカウント情報を保持するクラスです。
 *
 * @details
 * PairingAccountInfo のインスタンスが有効な状態であっても、
 * 特定の状態変化が発生するとデータの取得ができなくなります。
 * データが取得できなくなる条件については @ref PairingInfo の説明をご覧ください。
 */
class PairingAccountInfo
{
public:
    /**
     * @brief 中身が無効である状態のインスタンスを作成します。
     */
    PairingAccountInfo() NN_NOEXCEPT;

    /**
     * @brief インスタンスが保持するデータを破棄して無効な状態にします。
     */
    ~PairingAccountInfo() NN_NOEXCEPT;

    /**
     * @brief インスタンスが保持するアカウントの状態を取得します。
     * @param[out] outState アカウントの状態を受け取るポインター
     * @return 状態が取得できた場合は true
     * @pre
     *  - outState != nullptr
     *  - このインスタンスの状態が有効である
     *
     * @details
     * *outState の値が @ref PairingAccountState_Active 以外である場合、
     * 本クラスの他のメソッドから得られる情報が意図しない情報になる可能性があります。
     *
     * 戻り値が false であった場合はデータが取得できない状態になっています。
     * その場合、@ref PairingInfo の説明内容に従い
     * インスタンスを取得し直してください。
     */
    bool GetAccountState(PairingAccountState* outState) NN_NOEXCEPT;

    /**
     * @brief アカウントのニックネームを取得します。
     * @param[out] pNickname ニックネームを受け取るバッファー
     * @param[in] nicknameSize pNickname が指すバッファーのサイズ(バイト数)。
     * @return pNickname に書き込まれたバイト数(終端文字含む)または 0。
     *     0 であった場合はデータが取得できない状態になっています。
     * @pre
     *  - このインスタンスの状態が有効である
     *
     * @details
     * ニックネームの最大長はニンテンドーアカウントの仕様に準じます。
     * (終端文字を除いて nn::account::NicknameBytesMax となります。)
     *
     * 戻り値が 0 であった場合はデータが取得できない状態になっています。
     * その場合、@ref PairingInfo の説明内容に従い
     * インスタンスを取得し直してください。
     */
    size_t GetNickname(char* pNickname, size_t nicknameSize) NN_NOEXCEPT;

    /**
     * @brief アカウントのMii画像を取得します。
     * @param[out] pActualSize 画像データの実際の大きさ(バイト数)。
     * @param[out] pImage 画像データの格納先バッファー。
     * @param[in] bufferSize pImage が指すバッファーのサイズ(バイト数)
     * @return 処理の結果を表す nn::Result 値
     * @retval nn::ResultSuccess 有効な情報が返されました。
     * @retval nn::pctl::ResultNeedsRefresh データが取得できない状態になっています。
     * @pre
     *  - このインスタンスの状態が有効である
     *  - pActualSize != nullptr
     * @post
     *  - 戻り値が nn::ResultSuccess である場合
     *    - *pActualSize に画像サイズが格納されている
     *    - pImage != nullptr の場合
     *      - pImage に min(bufferSize, *pActualSize) のデータが格納されている
     *
     * @details
     * 画像のフォーマットはニンテンドーアカウントの仕様に準じます。
     * @ref GetMiiImageContentType メソッドで具体的な形式を取得できます。
     * なお、画像がない場合は *pActualSize に 0 が返されます。
     * (戻り値は true になります。)
     *
     * 画像は一時ファイルにダウンロードされますが、まだダウンロードされていない場合は
     * 本メソッド内でその処理を行うため呼び出し元をブロックします。
     * ダウンロードされる画像は @ref GetMiiImageContentType メソッドで取得する画像と同一であるため、
     * GetMiiImageContentType メソッドで取得済みである場合は本メソッドでのダウンロード処理は省略されます。
     * (ただしその場合でも事前の通信利用要求は必要です。)
     *
     * pImage に nullptr、または bufferSize に 0 を指定すると、
     * 本メソッドは pActualSize へ実際の大きさを返す処理のみを行います。
     *
     * 戻り値が nn::pctl::ResultNeedsRefresh であった場合は
     * データが取得できない状態になっています。その場合、@ref PairingInfo の説明内容に従い
     * インスタンスを取得し直してください。
     */
    nn::Result GetMiiImage(size_t* pActualSize, void* pImage, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief アカウントのMii画像の形式を示す文字列(type/subtype 形式)を取得します。
     * @param[out] pActualLength 実際のContent-Type文字列の長さ(終端のNULL文字は含みません)。
     * @param[out] outContentType Content-Type文字列の格納先バッファー。
     * @param[in] maxLength outContentType が指すバッファーがNULL文字込みで保持できる最大文字列長
     * @return 処理の結果を表す nn::Result 値
     * @retval nn::ResultSuccess 有効な情報が返されました。
     * @retval nn::pctl::ResultNeedsRefresh データが取得できない状態になっています。
     * @pre
     *  - このインスタンスの状態が有効である
     *  - pActualLength != nullptr
     * @post
     *  - 戻り値が nn::ResultSuccess である場合
     *    - *pActualLength に文字列の長さが格納されている
     *    - outContentType != nullptr の場合
     *      - outContentType にNULL文字込みで min(maxLength, *pActualLength + 1) のデータが格納されている
     *
     * @details
     * 具体的に返される形式はニンテンドーアカウントの仕様に準じます。
     * なお、画像がない場合は *pActualLength に 0 が返され、
     * outContentType が nullptr でなければ outContentType は空文字列になります。
     * (戻り値は true になります。)
     *
     * 画像は一時ファイルにダウンロードされますが、まだダウンロードされていない場合は
     * 本メソッド内でその処理を行うため呼び出し元をブロックします。
     * ダウンロードされる画像は @ref GetMiiImage メソッドで取得する画像と同一であるため、
     * GetMiiImage メソッドで取得済みである場合は本メソッドでのダウンロード処理は省略されます。
     * (ただしその場合でも事前の通信利用要求は必要です。)
     *
     * outContentType に nullptr、または maxLength に 0 を指定すると、
     * 本メソッドは pActualLength へ実際の大きさを返す処理のみを行います。
     *
     * 戻り値が nn::pctl::ResultNeedsRefresh であった場合は
     * データが取得できない状態になっています。その場合、@ref PairingInfo の説明内容に従い
     * インスタンスを取得し直してください。
     */
    nn::Result GetMiiImageContentType(size_t* pActualLength, char* outContentType, size_t maxLength) NN_NOEXCEPT;

    /**
     * @brief このインスタンスが有効かどうかを返します。
     * @return インスタンスが有効である場合は true
     *
     * @details
     * 各種情報取得メソッドを呼び出すにはインスタンスが有効である必要があります。
     */
    bool IsInstanceValid() const NN_NOEXCEPT;

    /**
     * @brief (デバッグ用)このインスタンスが有効かどうかを返します。
     * @deprecated 本メソッドは非デバッグ用途でも利用できる IsInstanceValid() に置き換えられました。
     * @return インスタンスが有効である場合は true
     *
     * @details
     * 各種情報取得メソッドを呼び出すにはインスタンスが有効である必要があります。
     */
    NN_DEPRECATED bool DebugIsInstanceValid() const NN_NOEXCEPT
    {
        return IsInstanceValid();
    }

private:
    uint8_t m_Reserved[16];
};

/**
 * @brief 連携先の相手の情報を保持するクラスです。
 *
 * @details
 * PairingInfo を使用したデータの取得は、以下に挙げる状態変化が発生すると
 * それ以上取得を行うことができなくなります。
 *
 *  - 新たに @ref RequestPairing を呼び出す
 *  - @ref RequestPairing が返す PairingInfo を用いて @ref AuthorizePairing を呼び出す
 *    - PairingInfo のデータが更新されるため、更新後のインスタンスを使用することは問題ありません。
 *      また、失敗した場合は以前のデータが引き続き利用できます。
 *  - 新たに @ref RetrievePairingInfo を呼び出す
 *  - @ref UnlinkPairing または @ref DeletePairing を呼び出す
 *  - 本体システムが再起動される
 *
 * これらの状態変化があった場合、PairingInfo (および PairingInfo から
 * 取得できる @ref PairingAccountInfo) のインスタンス自体は有効のままですが、
 * 各種取得関数が有効な情報を返さなくなります。
 * その場合、連携済みである場合は改めて @ref RetrievePairingInfo を、
 * 未連携である場合は再度 @ref RequestPairing を呼び出す必要があります。
 *
 * なお、バックグラウンドで自動的に連携が解除された場合、取得済みの
 * PairingInfo インスタンスには影響はありません。
 * (ただし @ref PairingAccountInfo::GetMiiImage での画像ダウンロードが
 * できなくなる可能性はあります。)
 */
class PairingInfo
{
public:
    /**
     * @brief 中身が無効である状態のインスタンスを作成します。
     */
    PairingInfo() NN_NOEXCEPT;

    /**
     * @brief インスタンスが保持するデータを解放し無効な状態にします。
     */
    ~PairingInfo() NN_NOEXCEPT;

    /**
     * @brief 連携先相手のアカウント情報を取得します。
     * @param[out] pAccountInfo アカウント情報を受け取るインスタンスのポインター
     * @return 有効な情報を返すことができたら true
     * @pre
     *  - このインスタンスの状態が有効である
     *  - @ref GetState() != @ref PairingState_None
     *  - pAccountInfo != nullptr
     * @post
     *  - 戻り値が true である場合
     *    - pAccountInfo のインスタンスの状態が有効である
     *
     * @details
     * pAccountInfo が指すインスタンスの状態が既に有効である場合、
     * pAccountInfo が保持していたインスタンスデータは上書きされます。
     *
     * 戻り値が false であった場合はデータが取得できない状態になっています。
     * その場合、@ref PairingInfo の説明内容に従い
     * インスタンスを取得し直してください。
     */
    bool GetAccountInfo(PairingAccountInfo* pAccountInfo) NN_NOEXCEPT;

    /**
     * @brief 現在の連携状態を返します。
     * @return 連携状態を表す @ref PairingState の各値
     * @pre
     *  - このインスタンスの状態が有効である
     */
    PairingState GetState() const NN_NOEXCEPT;

    /**
     * @brief (デバッグ用)このインスタンスが有効かどうかを返します。
     * @return インスタンスが有効である場合は true
     *
     * @details
     * 各種情報取得メソッドを呼び出すにはインスタンスが有効である必要があります。
     */
    bool IsInstanceValid() const NN_NOEXCEPT;

    /**
     * @brief (デバッグ用)このインスタンスが有効かどうかを返します。
     * @deprecated 本メソッドは非デバッグ用途でも利用できる IsInstanceValid() に置き換えられました。
     * @return インスタンスが有効である場合は true
     *
     * @details
     * 各種情報取得メソッドを呼び出すにはインスタンスが有効である必要があります。
     */
    NN_DEPRECATED bool DebugIsInstanceValid() const NN_NOEXCEPT
    {
        return IsInstanceValid();
    }

private:
    uint8_t m_Reserved[16];
};

}}
