﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/pctl/pctl_TypesSystem.h>
#include <nn/pctl/pctl_ResultSystem.h>

#include <nn/pctl/pctl_ApiSystemReadOnly.h>

namespace nn { namespace pctl {

/**
 * @name 本体機能(システム)向けペアコン状態チェック関数
 * @{
 */

/**
 * @brief ペアコンの制限設定に基づいてアプリケーション起動が可能かどうかを確認します。
 * @param[in] applicationId 起動したいアプリケーションのアプリケーションID
 * @param[in] controlProperty 起動したいアプリケーションの情報を持った nn::ns::ApplicationControlProperty データ
 * @return nn::Result の値
 * @retval nn::ResultSuccess アプリケーションの起動が可能です(制限されていないか一時解除状態です)
 * @retval nn::pctl::ResultRestrictedByRating レーティング制限によりアプリケーション起動が制限されています
 *
 * @details
 * 本関数は一時解除状態であるかどうかも加味した値を返します。
 * 一時解除状態である場合は nn::ResultSuccess が返ります。
 */
nn::Result ConfirmLaunchApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT;

/**
 * @brief ペアコンの制限設定に基づいて中断されているアプリケーションの再開が可能かどうかを確認します。
 * @param[in] applicationId 中断されているアプリケーションのアプリケーションID
 * @param[in] controlProperty 中断されているアプリケーションの情報を持った nn::ns::ApplicationControlProperty データ
 * @return nn::Result の値
 * @retval nn::ResultSuccess アプリケーションの再開が可能です(制限されていないか一時解除状態です)
 * @retval nn::pctl::ResultFreeCommunicationRestricted 「他の人との自由なコミュニケーション」機能の利用が制限されています
 * @retval nn::pctl::ResultRestrictedByRating レーティング制限によりアプリケーション起動が制限されています
 * @retval nn::pctl::ResultStereoVisionRestricted 立体視機能の利用が制限されています
 *
 * @details
 * 本関数は中断されているアプリケーションを再開する際に必要なチェックを行います。
 * アプリケーション起動時のチェック(@ref ConfirmLaunchApplicationPermission 関数を利用)とは
 * 一部制限が発生する条件が異なります。本関数は主に以下のような処理を行います。
 *
 * - アプリケーションが「他の人との自由なコミュニケーション」機能および立体視機能を利用していない場合は、
 *   ConfirmLaunchApplicationPermission とほぼ同様レーティングによる制限チェックを行い、
 *   その結果に基づく戻り値を返します。
 * - アプリケーションが「他の人との自由なコミュニケーション」機能または立体視機能を利用した場合は、
 *   レーティングによる制限に加えてこれらの機能の制限チェックも行います。
 */
nn::Result ConfirmResumeApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT;

/**
 * @brief 制限を一時的に解除した状態をリセットし、解除していない状態にします。
 * @post
 *  - @ref IsRestrictionTemporaryUnlocked の戻り値が false になる
 *
 * @details
 * 一時解除状態である場合は CheckFreeCommunicationPermission や
 * @ref ConfirmSnsPostPermission などが成功を返します。
 */
void RevertRestrictionTemporaryUnlocked() NN_NOEXCEPT;

/**
 * @brief 汎用設定利用・変更の許可が必要なシステムのシーンに入ったことを記録します。
 * @post
 *  - @ref IsRestrictedSystemSettingsEntered が true を返すようになる
 *
 * @details
 * 内部でシーンに入った回数がカウントされており、本関数はその回数を増やします。
 * 該当のシーンに入る前に本関数を呼び出し、シーンを抜けた際に
 * @ref LeaveRestrictedSystemSettings を呼び出すようにしてください。
 *
 * 本関数は @ref ConfirmSystemSettingsPermission の結果に影響を与えません。
 */
void EnterRestrictedSystemSettings() NN_NOEXCEPT;

/**
 * @brief 汎用設定利用・変更の許可が必要なシステムのシーンから出たことを記録します。
 * @post
 *  - 内部のカウントが 0 になった場合 @ref IsRestrictedSystemSettingsEntered が false を返すようになる
 *
 * @details
 * 内部でシーンに入った回数がカウントされており、本関数はその回数を減らします。
 * 該当のシーンに入る前に @ref EnterRestrictedSystemSettings を呼び出し、
 * シーンを抜けた際に本関数を呼び出すようにしてください。@n
 * 何らかの理由でカウントが 0 の状態で本関数を呼び出した場合、
 * 本関数は何も行いません。
 *
 * 本関数は @ref ConfirmSystemSettingsPermission の結果に影響を与えません。
 */
void LeaveRestrictedSystemSettings() NN_NOEXCEPT;

/**
 * @brief 汎用設定利用・変更の許可が必要なシステムのシーンに入っている状態をリセットし、
 *        シーンに入る前と同等の状態にします。
 * @post
 *  - @ref IsRestrictedSystemSettingsEntered が false を返すようになる
 *
 * @details
 * 既に @ref IsRestrictedSystemSettingsEntered が false を返す状態
 * (内部で保持しているカウントが 0 の状態)で本関数を呼び出しても何も起きず、
 * 内部状態は変わりません。
 *
 * 本関数は @ref ConfirmSystemSettingsPermission の結果に影響を与えません。
 */
void RevertRestrictedSystemSettingsEntered() NN_NOEXCEPT;

/**
 * @}
 */
/**
 * @name ペアコン設定変更関数
 * @{
 */

/**
 * @brief ペアコン設定の安心レベルを設定します。
 * @param[in] level 安心レベル
 * @pre
 *  - level が有効な @ref SafetyLevel の値である
 * @post
 *  - @ref GetCurrentSettings で取得できる値が安心レベル設定に基づいたものになる
 *
 * @details
 * 安心レベルを @ref SafetyLevel_Custom 以外に設定した場合、
 * SafetyLevel_Custom 以外のレベルで使用される設定値(プリセット値)が
 * 各種制限を行う際に利用されます。(@ref ConfirmLaunchApplicationPermission 、
 * @ref ConfirmResumeApplicationPermission 、@ref ConfirmSnsPostPermission 、
 * @ref ConfirmSystemSettingsPermission 、CheckFreeCommunicationPermission の各関数に影響します。)@n
 * この場合、アプリケーション別の「他の人との自由なコミュニケーション」機能に対する制限の値は
 * プリセット値内の既定値が利用されます。
 *
 * 安心レベルを設定すると、@ref GetCurrentSettings で取得できる値が更新されます。
 * GetCurrentSettings で取得できる値が @ref GetSafetyLevelSettings で取得できる
 * 値と異なるものになっていた場合、更新前の値は上書きされます。
 *
 * 本関数は一時解除コードの設定を行う前に呼び出すことができ、
 * その値は @ref GetSafetyLevel 関数などで取得することが可能ですが、
 * 一時解除コードが設定されていない状態では設定された値が制限判定に用いられることはありません。
 */
void SetSafetyLevel(SafetyLevel level) NN_NOEXCEPT;

/**
 * @brief ペアコン設定の安心レベルに紐付く既定のペアコン設定(プリセット値)を取得します。
 * @param[out] pSettings プリセット値を受け取るポインター
 * @param[in] level 安心レベル
 * @pre
 *  - pSettings != nullptr
 *  - level が有効な @ref SafetyLevel の値である
 *
 * @details
 * level に @ref SafetyLevel_Custom を指定した場合はカスタムで設定した値が取得できます。
 * level が SafetyLevel_Custom ではない場合は、本体のリージョン設定によって
 * 取得される値が変わる場合があります。
 *
 * level が SafetyLevel_Custom ではない場合に本関数が返す値は「既定値」です。
 * この値は本体の更新などによって変更される場合があります。
 * 一方、@ref GetCurrentSettings で取得される値は @ref GetSafetyLevel が SafetyLevel_Custom 以外を
 * 返す場合であっても「既定値」ではなくそれまで設定されていた値を返します。
 * そのため、本関数で取得される値は GetCurrentSettings で取得できる値と
 * 異なる値になる可能性があります。
 *
 * ペアコンが一度も設定されていない場合(システム初期状態から、
 * または @ref DeleteSettings が呼び出された状態から一度も
 * @ref SetCustomSafetyLevelSettings が呼び出されていない場合)、
 * 本関数でカスタム設定の取得を行うとすべて 0 に相当する設定値を返します。
 */
void GetSafetyLevelSettings(SafetyLevelSettings* pSettings, SafetyLevel level) NN_NOEXCEPT;

/**
 * @brief 安心レベルが「カスタム」(@ref SafetyLevel_Custom)に設定されている場合に利用する、
 *   アプリケーション別の「他の人との自由なコミュニケーション」機能の制限を除く各種制限値を設定します。
 * @param[in] settings 各種制限値を持つデータ
 * @pre
 *  - settings.ratingAge >= 0 && settings.ratingAge <= 255
 * @post
 *  - @ref GetSafetyLevelSettings に SafetyLevel_Custom を指定した場合に取得できる値が
 *    本関数で指定した値になる
 *  - @ref GetSafetyLevel() == SafetyLevel_Custom である場合
 *    - @ref GetCurrentSettings で取得できる値が本関数で設定した値になる
 *
 * @details
 * 本関数で設定される「他の人との自由なコミュニケーション」機能の制限は
 * 未知のアプリケーションに対する初期の設定値になります。
 *
 * @ref GetSafetyLevel の戻り値が SafetyLevel_Custom ではない場合は、
 * 現在のペアコン設定としては反映されず、@ref GetCurrentSettings の結果にも影響を与えません。
 *
 * 本関数は一時解除コードの設定を行う前に呼び出すことができ、
 * その値は @ref GetSafetyLevelSettings などで取得することが可能ですが、
 * 一時解除コードが設定されていない状態では設定された値が制限判定に用いられることはありません。
 */
void SetCustomSafetyLevelSettings(const SafetyLevelSettings& settings) NN_NOEXCEPT;

/**
 * @brief レーティング設定による制限を行う際に利用する既定のレーティング団体を取得します。
 * @return レーティング団体を表す値
 *
 * @details
 * 本関数は @ref SetDefaultRatingOrganization に基づいて設定された値を返します。
 * また、SetDefaultRatingOrganization による設定が行われていない状態でこの関数を呼び出すと、
 * リージョンと言語からあらかじめ決められたパターンに沿って使用するレーティング団体を
 * 本関数の戻り値として返します。
 */
nn::ns::RatingOrganization GetDefaultRatingOrganization() NN_NOEXCEPT;

/**
 * @brief レーティング設定による制限を行う際に利用する既定のレーティング団体を設定します。
 * @param[in] organization レーティング団体を表す値
 *
 * @details
 * 本関数は一時解除コードの設定を行う前に呼び出すことができますが、
 * 一時解除コードが設定されていない状態では設定された値が制限判定に用いられることはありません。
 */
void SetDefaultRatingOrganization(nn::ns::RatingOrganization organization) NN_NOEXCEPT;

/**
 * @brief 現在設定されているペアコンの制限情報をすべて削除し、制限のない状態にします。
 * @post
 *  - @ref IsRestrictionEnabled() == false
 *
 * @details
 * - 設定されていた解除コードも削除されます。
 * - @ref GetFreeCommunicationApplicationList で得られるリストについては
 *   設定値のみリセットされ、各要素自体は削除されません。
 *   同様に、ローカルに保存された制限対象外リストも、設定値のみ既定値にリセットされます。
 * - 削除した値は復元できません。
 * - @ref IsRestrictionTemporaryUnlocked や @ref IsRestrictedSystemSettingsEntered の戻り値も
 *   false にリセットされます。
 */
void DeleteSettings() NN_NOEXCEPT;

/**
 * @brief ペアコンの制限処理をすべて無効化します。
 *
 * @details
 * - 本関数は、本体初期化などすべての設定を削除する際に、バックグラウンドで行う
 *   ペアコン関連処理によってデータが削除後に書き出されてしまうのを
 *   防止するために利用します。
 * - 本関数を呼び出しても設定値は削除されず、無効化した状態のまま
 *   本体を再起動すると再び有効な状態になります。
 *   (無効化状態のまま DeletaPairing や DeleteSettings、およびシステムによる
 *   システムセーブデータ削除処理などを行うと設定値も削除されます。)
 * - サーバー連携を行っている場合、それに伴う処理もすべて一時無効化されます。
 * - 既に無効化されている状態で本関数を呼び出しても何も行いません。
 */
void DisableFeaturesForReset() NN_NOEXCEPT;

/**
 * @brief 「他の人との自由なコミュニケーション」の機能を有するアプリケーションの総数を返します。
 * @return アプリケーションの数
 *
 * @details
 * 本関数が返すアプリケーションの数は、呼び出し時点における @ref GetFreeCommunicationApplicationList で
 * 得られるデータの総数に一致します。
 *
 * @ref AddToFreeCommunicationApplicationList の非同期処理が実行中である場合、
 * 本関数はその処理(数10ms程度)が終わるまで呼び出し元をブロックします。
 */
int GetFreeCommunicationApplicationListCount() NN_NOEXCEPT;

/**
 * @brief アプリケーション別の「他の人との自由なコミュニケーション」に関する利用制限データのリストを取得します。
 * @param[out] pOutInfoArray アプリケーション別の利用制限データを受け取るための配列
 * @param[in] offset データの取得を開始する内部配列のオフセット
 * @param[in] count pOutInfoArray で受け取れる最大の要素数
 * @return pOutInfoArray に出力された設定値の数(要素数)
 * @pre
 *  - pOutInfoArray != nullptr
 *  - offset >= 0
 *  - count >= 0
 * @post
 *  - 戻り値 <= count
 *  - 戻り値が 0 より大きい場合、pOutInfoArray は 0 から (戻り値 - 1) のインデックスに対応する要素が有効である
 *
 * @details
 * offset から数えた内部配列の要素数が count より小さい場合、本関数の戻り値は count より少なくなります。
 * 特に、offset が内部配列の全体の要素数以上になっている場合、戻り値は 0 となります。
 *
 * @ref AddToFreeCommunicationApplicationList の非同期処理が実行中である場合、
 * 本関数はその処理(数10ms程度)が終わるまで呼び出し元をブロックします。
 */
int GetFreeCommunicationApplicationList(FreeCommunicationApplicationInfo* pOutInfoArray, int offset, int count) NN_NOEXCEPT;

/**
 * @brief アプリケーション別の「他の人との自由なコミュニケーション」に関する利用制限のリストを更新します。
 * @param[in] pInfoArray アプリケーション別の利用制限データを保持する配列
 * @param[in] count pInfoArray の要素数
 * @pre
 *  - pInfoArray != nullptr
 *  - count > 0
 *  - pInfoArray の各要素はすべて内部配列に登録されているアプリケーションである
 *    (@ref GetFreeCommunicationApplicationList で取得されるリストに含まれるアプリケーションである)
 *
 * @details
 * 本関数に指定する pInfoArray の要素数は必ずしも @ref GetFreeCommunicationApplicationListCount で
 * 得られる数と一致する必要はありません。内部配列に存在するものの pInfoArray に
 * 含まれていないアプリケーションに対しては、設定値の削除は行われずそのまま内部配列に残ります。
 *
 * pInfoArray の要素として同じアプリケーションに対する設定値が複数含まれていた場合の挙動は未定義です。
 *
 * @note
 * 本関数は設定値をストレージへ書き込む処理を行いますが、
 * この書き込み処理は並列(非同期)で行うため、その処理の間ブロックすることはありません。
 */
void UpdateFreeCommunicationApplicationList(const FreeCommunicationApplicationInfo* pInfoArray, int count) NN_NOEXCEPT;

/**
 * @brief アプリケーション別の「他の人との自由なコミュニケーション」の利用制限リストにアプリケーションを追加します。
 * @param[in] applicationId アプリケーションのアプリケーションID
 * @param[in] controlProperty アプリケーションに対するアプリケーションの情報
 * @pre
 *  - applicationId != nn::ncm::ApplicationId::GetInvalidId()
 *
 * @details
 * 本関数は controlProperty の内容を確認し、必要に応じてシステム内部で管理する配列に追加します。
 * アプリケーションが「他の人との自由なコミュニケーション」の機能を有していない(制限が不要である)か、
 * 既に内部配列にアプリケーションが含まれている場合は何も処理を行いません。
 *
 * メモリ上の内部配列への追加処理は同期的に行われるため、本関数は
 * その処理の間呼び出し元をブロックします。
 *
 * @note
 * - 本関数はIPCを行う前に controlProperty の内容を確認し、その段階で制限が不要であると
 *   判断した場合は、IPCを行わずに処理を終了します。
 * - リスト追加への同期処理は内部のメモリ上のリストに追加する処理がメインとなります。
 *   ストレージへの反映(書き込み)は並列に行うため、その処理の間
 *   追加で呼び出し元をブロックすることはありません。
 */
void AddToFreeCommunicationApplicationList(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT;

/**
 * @brief アプリケーションのダウンロードが開始されたことを pctl に通知します。
 * @param[in] applicationId アプリケーションのアプリケーションID
 * @pre
 *  - applicationId != nn::ncm::ApplicationId::GetInvalidId()
 *
 * @details
 * 本関数は、システム内で「ダウンロードが開始された」というイベントを残すために使用されます。
 *
 * 本機能はメンテナンスモードでも呼び出すことが可能です(何も処理を行いません)。
 */
void NotifyApplicationDownloadStarted(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @}
 */

}}
