﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/detail/service/common/pctl_Cancelable.h>
#include <nn/nn_Result.h>
#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

/*!
    @brief      RapidJSON 用の入力ストリームです。

    @details
                RapidJSON のパース処理で指定する InputStream のメソッドを実装しています。@n
                入力ソースの Open/Close、及び、データの読み込み処理（FillImpl）は本クラスを継承して記述する必要があります。

                本クラスは作業用バッファを持たないため、利用時に @ref SetBuffer でバッファを指定する必要があります。
*/
class JsonInputStream
{
public:
    typedef char Ch;

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonInputStream() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    virtual ~JsonInputStream() NN_NOEXCEPT;

    /*!
        @brief      バッファを設定します。

        @param[in]  stringBuffer        文字列値を格納するバッファ。
        @param[in]  stringBufferSize    文字列値を格納するバッファのサイズ。
        @param[in]  streamBuffer        入力値を格納するバッファ。
        @param[in]  streamBufferSize    入力値を格納するバッファのサイズ。

        @pre
            - stringBuffer != nullptr
            - stringBufferSize >= 8
            - streamBuffer != nullptr
            - streamBufferSize >= 8
    */
    void SetBuffer(Ch* stringBuffer, size_t stringBufferSize, Ch* streamBuffer, size_t streamBufferSize) NN_NOEXCEPT;

    /*!
        @brief      キャンセル可能オブジェクトを設定します。

        @param[in]  cancelable  キャンセル可能オブジェクト。

        @details
                    キャンセル可能オブジェクトを設定することで、入力処理を中断することが可能です。
    */
    void SetCancelable(const common::Cancelable* cancelable) NN_NOEXCEPT;

    /*!
        @brief      オーバーフロー検出器を取得します。

        @return     オーバーフロー検出器。
    */
    const std::atomic<bool>* GetOverflowDetector() NN_NOEXCEPT;

    /*!
        @brief      入力中に発生した最後のエラーを取得します。

        @return     入力中に発生した最後のエラー。
    */
    nn::Result GetLastError() const NN_NOEXCEPT;

public:
    /*!
        @brief      読み込み位置を移動せずに入力ストリームの先頭文字を取得します。

        @return     先頭文字。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    Ch Peek() NN_NOEXCEPT;

    /*!
        @brief      入力ストリームの先頭文字を取得します。

        @return     先頭文字。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    Ch Take() NN_NOEXCEPT;

    /*!
        @brief      読み込み位置を取得します。

        @return     読み込み位置。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    size_t Tell() const NN_NOEXCEPT;

    /*!
        @brief      文字列値のパースを開始します。

        @return     文字列バッファの先頭。

        @details
                    本関数は、RapidJSON 用関数です。@n
                    kParseInsituFlag を指定したパースでは、本関数を使用して文字列バッファの初期化を行います。
    */
    Ch* PutBegin() NN_NOEXCEPT;

    /*!
        @brief      文字列値を入力します。

        @param[in]  c   文字。

        @details
                    本関数は、RapidJSON 用関数です。@n
                    kParseInsituFlag を指定したパースでは、本関数を使用して文字列を入力します。
    */
    void Put(Ch c) NN_NOEXCEPT;

    /*!
        @brief      文字列値のパースを終了します。

        @param[in]  begin   文字列バッファの先頭。

        @return     文字列の長さ。

        @pre
            - begin != PutBegin()

        @details
                    本関数は、RapidJSON 用関数です。
    */
    size_t PutEnd(Ch* begin) NN_NOEXCEPT;

protected:
    /*!
        @brief      入力バッファにデータを書き込みます。

        @param[out] outRead     読み込んだバイト数。
        @param[in]  buffer      入力バッファ。
        @param[in]  bufferSize  入力バッファサイズ。

        @return     処理結果。

        @details
                    本関数は実際のデータ取得処理を行うため、時間がかかる可能性があります。@n
                    本関数を実装する際、適度なタイミングで @ref JsonInputStream::IsCanceled でキャンセル状態を確認することを推奨します。
    */
    virtual nn::Result FillImpl(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT = 0;

    /*!
        @brief      キャンセルされたかどうかを判定します。

        @return     キャンセルされたかどうか。

        @details
                    @ref SetCancelable でキャンセル可能オブジェクトを設定していた場合、 Cancelable::IsCanceled を呼び出します。@n
                    キャンセル可能オブジェクトが設定されていない場合、常に false を返します。
    */
    bool IsCanceled() const NN_NOEXCEPT;

private:
    //
    Ch* m_String;
    size_t m_StringBufferSize;
    //
    Ch* m_Stream;
    size_t m_StreamBufferSize;
    //
    size_t m_StringSize;
    //
    size_t m_Read;
    size_t m_ReadTotal;
    size_t m_PositionInStream;
    //
    nn::Result m_LastError;
    //
    const common::Cancelable* m_Cancelable;
    //
    std::atomic<bool> m_OverflowDetector;

private:
    /*!
        @brief      入力バッファにデータを書き込みます。

        @return     1 バイト以上入力があったかどうか。

        @details
                    本関数は @ref InputStream::FillImpl を呼び出し、共通のハンドリング処理を行います。
    */
    bool Fill() NN_NOEXCEPT;
};

/*!
    @brief      メモリ上のデータを読み込む入力ストリームです。
*/
class JsonMemoryInputStream : public JsonInputStream
{
public:
    typedef JsonInputStream::Ch Ch;

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonMemoryInputStream() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    virtual ~JsonMemoryInputStream() NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを開きます。

        @param[in]  memory  入力データ。
        @param[in]  size    入力データのサイズ。

        @pre
            - memory != nullptr
            - size > 0

        @return     処理結果。
    */
    nn::Result Open(const void* memory, size_t size) NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを閉じます。
    */
    void Close() NN_NOEXCEPT;

protected:
    //
    virtual nn::Result FillImpl(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    //
    const void* m_Memory;
    size_t m_Size;
    //
    bool m_IsOpened;
    //
    size_t m_Position;
};

}}}}}
