﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   Standard PCI definitions (C++ language)
 */

namespace nn { namespace pcie {


/**
 * @brief Size of standard PCI configuration header
 */
const size_t  StandardConfigHeaderSize       = 64;

/**
 * @brief Size of standard PCI configuration space
 */
const size_t  StandardConfigSpaceSize        = 256;

/**
 * @brief Size of standard PCIe configuration space
 */
const size_t  StandardConfigSpaceExpressSize = 4096;

/**
 * @brief Maximum number of base address registers, used for
 *        mapping registers implemented on device
 */
const int32_t StandardConfigMaxBaseAddresses = 6;

/**
 * @brief Registers offsets of standard PCI configuration space
 */
enum StandardConfigOffset
{
    StandardConfigOffset_VendorId           = 0x00,
    StandardConfigOffset_DeviceId           = 0x02,
    StandardConfigOffset_Command            = 0x04,
    StandardConfigOffset_Status             = 0x06,        // 16 bits
    StandardConfigOffset_ClassRevision      = 0x08,        // High 24 bits are class, low 8 revision
    StandardConfigOffset_RevisionId         = 0x08,        // Revision ID
    StandardConfigOffset_ClassProg          = 0x09,        // Reg. Level Programming Interface
    StandardConfigOffset_ClassDevice        = 0x0a,        // Device class
    StandardConfigOffset_CacheLineSize      = 0x0c,        // 8-bit
    StandardConfigOffset_LatencyTimer       = 0x0d,        // 8-bit
    StandardConfigOffset_HeaderType         = 0x0e,        // 8-bit
    StandardConfigOffset_Bist               = 0x0f,        // 8-bit
    StandardConfigOffset_BaseAddress0       = 0x10,        // 32-bit
    StandardConfigOffset_BaseAddress1       = 0x14,        // 32-bit, htype 0,1 only
    StandardConfigOffset_BaseAddress2       = 0x18,        // 32-bit, htype 0 only
    StandardConfigOffset_BaseAddress3       = 0x1C,        // 32-bit
    StandardConfigOffset_BaseAddress4       = 0x20,        // 32-bit
    StandardConfigOffset_BaseAddress5       = 0x24,        // 32-bit
};

/**
 * @brief Registers values within standard PCI configuration space
 */
enum StandardConfigValue
{
    StandardConfigValue_CommandIo              = 0x1,        // Enable response in I/O space
    StandardConfigValue_CommandMemory          = 0x2,        // Enable response in Memory space
    StandardConfigValue_CommandMaster          = 0x4,        // Enable bus mastering
    StandardConfigValue_CommandSpecial         = 0x8,        // Enable response to special cycles
    StandardConfigValue_CommandInvalidate      = 0x10,       // Use memory write and invalidate
    StandardConfigValue_CommandVgaPalette      = 0x20,       // Enable palette snooping
    StandardConfigValue_CommandParity          = 0x40,       // Enable parity checking
    StandardConfigValue_CommandWait            = 0x80,       // Enable address/data stepping
    StandardConfigValue_CommandSerr            = 0x100,      // Enable SERR
    StandardConfigValue_CommandFastBack        = 0x200,      // Enable back-to-back writes
    StandardConfigValue_CommandIntxDisable     = 0x400,      // INTx Emulation Disable

    StandardConfigValue_StatusInterrupt        = 0x08,      // Interrupt status
    StandardConfigValue_StatusCapabilityList   = 0x10,      // Support Capability List
    StandardConfigValue_Status66Mhz            = 0x20,      // Support 66 MHz PCI 2.1 bus
    StandardConfigValue_StatusFastBack         = 0x80,      // Accept fast-back to back
    StandardConfigValue_StatusParity           = 0x100,     // Detected parity error
    StandardConfigValue_StatusDevselMask       = 0x600,     // DEVSEL timing
    StandardConfigValue_StatusDevselFast       = 0x000,
    StandardConfigValue_StatusDevselMedium     = 0x200,
    StandardConfigValue_StatusDevselSlow       = 0x400,
    StandardConfigValue_StatusSigTargetAbort   = 0x800,     // Set on target abort
    StandardConfigValue_StatusRecTargetAbort   = 0x1000,    // Master ack of abort
    StandardConfigValue_StatusRecMasterAbort   = 0x2000,    // Set on master abort
    StandardConfigValue_StatusSigSystemError   = 0x4000,    // Set when we drive SERR
    StandardConfigValue_StatusDetectedParity   = 0x8000,    // Set on parity error

    StandardConfigValue_HeaderTypeNormal       = 0,
    StandardConfigValue_HeaderTypeBridge       = 1,
    StandardConfigValue_HeaderTypeCardbus      = 2,

    StandardConfigValue_BistCodeMask           = 0x0f,       // Return result
    StandardConfigValue_BistStart              = 0x40,       // 1 to start BIST, 2 secs or less
    StandardConfigValue_BistCapable            = 0x80,       // 1 if BIST capable
    StandardConfigValue_BaseAddressSpace       = 0x01,       // 0 = memory, 1 = I/O
    StandardConfigValue_BaseAddressSpaceIo     = 0x01,
    StandardConfigValue_BaseAddressSpaceMem    = 0x00,
    StandardConfigValue_BaseAddressMemTypeMask = 0x06,
    StandardConfigValue_BaseAddressMemType32   = 0x00,       // 32 bit address
    StandardConfigValue_BaseAddressMemType1M   = 0x02,       // Below 1M [obsolete]
    StandardConfigValue_BaseAddressMemType64   = 0x04,       // 64 bit address
    StandardConfigValue_BaseAddressMemPrefetch = 0x08,       // prefetchable?
    StandardConfigValue_MemMasked              = 0xfffffff0,
    StandardConfigValue_IoMasked               = 0xfffffffc,
};

/**
 * @brief Typical PCI devices of type '0'
 *        maximum number of base address registers
 */
const int32_t Type0ConfigMaxBaseAddresses = 6;

/**
 * @brief Typical PCI devices of type '0'
 *        configuration space register offsets
 */
enum Type0ConfigOffset
{
    Type0ConfigOffset_CardbusCis              = 0x28,
    Type0ConfigOffset_SubsystemVendorId       = 0x2c,
    Type0ConfigOffset_SubsystemId             = 0x2e,
    Type0ConfigOffset_RomAddress              = 0x30,
    Type0ConfigOffset_CapabilityList          = 0x34,   // Offset of first capability list entry
    Type0ConfigOffset_InterruptLine           = 0x3c,   // 8-bit
    Type0ConfigOffset_InterruptPin            = 0x3d,   // 8-bit
    Type0ConfigOffset_MinGnt                  = 0x3e,   // 8-bit
    Type0ConfigOffset_Lat                     = 0x3f,   // 8-bit
};

/**
 * @brief Typical PCI devices of type '0'
 *        configuration space register values
 */
enum Type0ConfigValue
{
    Type0ConfigValue_RomAddressEnable          = 0x01,
    Type0ConfigValue_RomAddressMask            = 0xfffff800,
    Type0ConfigValue_IntA                      = 1,
    Type0ConfigValue_IntB                      = 2,
    Type0ConfigValue_IntC                      = 3,
    Type0ConfigValue_IntD                      = 4,
};

/**
 * @brief PCI-to-PCI bridges of type '1'
 *        maximum number of base address registers
 */
const int32_t Type1ConfigMaxBaseAddresses = 2;

/**
 * @brief PCI-to-PCI bridges of type '1'
 *        configuration space register offsets
 */
enum Type1ConfigOffset
{
    Type1ConfigOffset_PrimaryBusNumber      = 0x18,      // Primary bus number
    Type1ConfigOffset_SecondaryBusNumber    = 0x19,      // Secondary bus number
    Type1ConfigOffset_SubordinateBusNumber  = 0x1a,      // Highest bus number behind the bridge
    Type1ConfigOffset_LatencyTimer          = 0x1b,      // Latency timer for secondary interface
    Type1ConfigOffset_IoBase                = 0x1c,      // I/O range behind the bridge
    Type1ConfigOffset_IoLimit               = 0x1d,
    Type1ConfigOffset_SecStatus             = 0x1e,      // Secondary status register, only bit 14 used
    Type1ConfigOffset_MemoryBase            = 0x20,      // Memory range behind
    Type1ConfigOffset_MemoryLimit           = 0x22,
    Type1ConfigOffset_PrefetchMemoryBase    = 0x24,      // Prefetchetchable memory range behind
    Type1ConfigOffset_PrefetchMemoryLimit   = 0x26,
    Type1ConfigOffset_PrefetchBaseUpper32   = 0x28,      // Upper half of prefetchable memory range
    Type1ConfigOffset_PrefetchLimitUpper32  = 0x2C,
    Type1ConfigOffset_IoBaseUpper16         = 0x30,      // Upper half of I/O addresses
    Type1ConfigOffset_IoLimitUpper16        = 0x32,
    // 0x34 same as for htype 0
    // 0x35-0x3b is reserved
    Type1ConfigOffset_RomAddress1           = 0x38,     // Same as PCI_ROM_ADDRESS, but for htype 1
    // 0x3c-0x3d are same as for htype 0
    Type1ConfigOffset_BridgeControl         = 0x3e,
};

/**
 * @brief PCI-to-PCI bridges of type '1'
 *         configuration space register values
 */
enum Type1ConfigValue
{
    Type1ConfigValue_IoRangeTypeMask           = 0x0f,       // I/O bridging type
    Type1ConfigValue_IoRangeType16             = 0x00,
    Type1ConfigValue_IoRangeType32             = 0x01,
    Type1ConfigValue_IoRangeMask               = 0xfffffff0, // Standard 4K I/O windows
    Type1ConfigValue_Io1kRangeMask             = 0xfffffffc, // Intel 1K I/O windows
    Type1ConfigValue_MemoryRangeTypeMask       = 0x0000000f,
    Type1ConfigValue_MemoryRangeMask           = 0xfffffff0,
    Type1ConfigValue_PrefetchRangeTypeMask     = 0x0000000f,
    Type1ConfigValue_PrefetchRangeType32       = 0x00,
    Type1ConfigValue_PrefetchRangeType64       = 0x01,
    Type1ConfigValue_PrefetchRangeMask         = 0xfffffff0,
    Type1ConfigValue_BridgeControlParity       = 0x01,        // Enable parity detection on secondary interface
    Type1ConfigValue_BridgeControlSerr         = 0x02,        // The same for SERR forwarding
    Type1ConfigValue_BridgeControlIsa          = 0x04,        // Enable ISA mode
    Type1ConfigValue_BridgeControlVga          = 0x08,        // Forward VGA addresses
    Type1ConfigValue_BridgeControlMasterAbort  = 0x20,        // Report master aborts
    Type1ConfigValue_BridgeControlBusReset     = 0x40,        // Secondary bus reset
    Type1ConfigValue_BridgeControlFastBack     = 0x80,        // Fast Back2Back enabled on secondary interface
};

/**
 * @brief CardBus bridges of type '2'
 *        configuration space register offsets
 */
enum Type2ConfigOffset
{
    Type2ConfigOffset_CapabilityList    = 0x14,
    // 0x15 reserved
    Type2ConfigOffset_SecondaryStatus   = 0x16,    // Secondary status
    Type2ConfigOffset_PrimaryBus        = 0x18,    // PCI bus number
    Type2ConfigOffset_CardBusNumber     = 0x19,    // CardBus bus number
    Type2ConfigOffset_SubordinateBus    = 0x1a,    // Subordinate bus number
    Type2ConfigOffset_LatencyTimer      = 0x1b,    // CardBus latency timer
    Type2ConfigOffset_MemoryBase0       = 0x1c,
    Type2ConfigOffset_MemoryLimit0      = 0x20,
    Type2ConfigOffset_MemoryBase1       = 0x24,
    Type2ConfigOffset_MemoryLimit1      = 0x28,
    Type2ConfigOffset_IoBase0           = 0x2c,
    Type2ConfigOffset_IoBase0Hi         = 0x2e,
    Type2ConfigOffset_IoLimit0          = 0x30,
    Type2ConfigOffset_IoLimitHi         = 0x32,
    Type2ConfigOffset_IoBase1           = 0x34,
    Type2ConfigOffset_IoBase1Hi         = 0x36,
    Type2ConfigOffset_IoLimit1          = 0x38,
    Type2ConfigOffset_IoLimit1Hi        = 0x3a,
    // 0x3c-0x3d are same as for htype 0
    Type2ConfigOffset_BridgeControl     = 0x3e,
    Type2ConfigOffset_SubsystemVendorId = 0x40,
    Type2ConfigOffset_SubsystemId       = 0x42,
    Type2ConfigOffset_LegacyModeBase    = 0x44,    // 16-bit PC Card legacy mode base address (ExCa)
    // 0x48-0x7f reserved
};

/**
 * @brief CardBus bridges of type '2'
 *        configuration space register values
 */
enum Type2ConfigValue
{
    Type2ConfigValue_IoRangeMask         = 0xfffffffc,
    Type2ConfigValue_ControlParity       = 0x01,        // Similar to standard bridge control register
    Type2ConfigValue_ControlSerr         = 0x02,
    Type2ConfigValue_ControlIsa          = 0x04,
    Type2ConfigValue_ControlVga          = 0x08,
    Type2ConfigValue_ControlMasterAbort  = 0x20,
    Type2ConfigValue_ControlReset        = 0x40,        // CardBus reset
    Type2ConfigValue_Control16bIntb      = 0x80,        // Enable interrupt for 16-bit cards
    Type2ConfigValue_ControlPrefetchMem0 = 0x100,       // Prefetchetch enable for both memory regions
    Type2ConfigValue_ControlPrefetchMem1 = 0x200,
    Type2ConfigValue_ControlPostWrites   = 0x400,
};

/**
 * @brief Limit on maximum number of base capability IDs
 */
const int32_t CapabilityIdLimit = 48;

/**
 * @brief Base capability IDs
 */
enum CapabilityId
{
    CapabilityId_List                             = 0,
    CapabilityId_PowerManagement                  = 0x01,
    CapabilityId_AcceleratorGraphicsPort          = 0x02,
    CapabilityId_VitalProductData                 = 0x03,
    CapabilityId_SlotId                           = 0x04,
    CapabilityId_Msi                              = 0x05,
    CapabilityId_CompactPciHotSwap                = 0x06,
    CapabilityId_Pcix                             = 0x07,
    CapabilityId_HyperTransport                   = 0x08,
    CapabilityId_VendorSpeciic                    = 0x09,
    CapabilityId_DebugPort                        = 0x0A,
    CapabilityId_CompactPciCentralResourceControl = 0x0B,
    CapabilityId_StandardHotPlugController        = 0x0C,
    CapabilityId_BridgeSubsystemVendorDeviceId    = 0x0D,
    CapabilityId_Agp3                             = 0x0E,
    CapabilityId_SecureDevice                     = 0x0F,
    CapabilityId_Pcie                             = 0x10,
    CapabilityId_Msix                             = 0x11,
    CapabilityId_Sata                             = 0x12,
    CapabilityId_AdvancedFeatures                 = 0x13,
    CapabilityId_Max                              = CapabilityId_AdvancedFeatures,
};

/**
 * @brief Power management structure offsets
 */
enum PowerManagementOffset
{
    PowerManagementOffset_Capability      = 2,
    PowerManagementOffset_Control         = 4,
    PowerManagementOffset_PpbExtenstions  = 6,
    PowerManagementOffset_DataRegister    = 7,
};

/**
 * @brief Power management structure values
 */
enum PowerManagementValue
{
    PowerManagementValue_CapabilityVersionMask          = 0x0007,
    PowerManagementValue_CapabilityPmeClock             = 0x0008,  // PME clock required
    PowerManagementValue_CapabilityReserved             = 0x0010,  // Reserved field
    PowerManagementValue_CapabilityDsi                  = 0x0020,  // Device specific initialization
    PowerManagementValue_CapabilityAuxPower             = 0x01C0,  // Auxiliary power support mask
    PowerManagementValue_CapabilityD1                   = 0x0200,  // D1 power state support
    PowerManagementValue_CapabilityD2                   = 0x0400,  // D2 power state support
    PowerManagementValue_CapabilityPme                  = 0x0800,  // PME pin supported
    PowerManagementValue_CapabilityPmeMask              = 0xF800,  // PME Mask of all supported states
    PowerManagementValue_CapabilityPmeD0                = 0x0800,  // PME# from D0
    PowerManagementValue_CapabilityPmeD1                = 0x1000,  // PME# from D1
    PowerManagementValue_CapabilityPmeD2                = 0x2000,  // PME# from D2
    PowerManagementValue_CapabilityPmeD3                = 0x4000,  // PME# from D3 (hot)
    PowerManagementValue_CapabilityPmeD3Cold            = 0x8000,  // PME# from D3 (cold)
    PowerManagementValue_CapabilityPmeShift             = 11,

    PowerManagementValue_ControlStateMask               = 0x0003,  // Current power state (D0 to D3)
    PowerManagementValue_ControlNoSoftReset             = 0x0008,  // No reset for D3hot->D0
    PowerManagementValue_ControlPmeEnable               = 0x0100,  // PME pin enable
    PowerManagementValue_ControlDataSelMask             = 0x1e00,  // Data select
    PowerManagementValue_ControlDataScaleMask           = 0x6000,  // Data scale
    PowerManagementValue_ControlPmeStatus               = 0x8000,  // PME pin status
    PowerManagementValue_ControlPpbB2B3                 = 0x40,    // Stop clock when in D3hot
    PowerManagementValue_ControlBpccEnable              = 0x80,    // Bus power/clock control enable
};

/**
 * @brief Message Signalled Interrupt (MSI) configuration space offsets
 */
enum MsiOffset
{
    MsiOffset_Flags        = 2,       // Message Control
    MsiOffset_Rfu          = 3,       // Rest of capability flags
    MsiOffset_AddressLo    = 4,       // Lower 32 bits
    MsiOffset_AddressHi    = 8,       // Upper 32 bits (if MsiOffset_Flags_64BIT set)
    MsiOffset_Data32       = 8,       // 16 bits of data for 32-bit devices
    MsiOffset_Mask32       = 12,      // Mask bits register for 32-bit devices
    MsiOffset_Pending32    = 16,      // Pending intrs for 32-bit devices
    MsiOffset_Data64       = 12,      // 16 bits of data for 64-bit devices
    MsiOffset_Mask64       = 16,      // Mask bits register for 32-bit devices
    MsiOffset_Pending64    = 20,      // Pending intrs for 64-bit devices
};

/**
 * @brief Message Signalled Interrupt (MSI) configuration space values
 */
enum MsiValue
{
    MsiValue_FlagsEnable     = 0x0001,  // MSI feature enabled
    MsiValue_FlagsQMask      = 0x000e,  // Maximum queue size available
    MsiValue_FlagsQSize      = 0x0070,  // Message queue size configured
    MsiValue_FlagsQSizeShift = 4,       // Message queue size bit position
    MsiValue_Flags64Bit      = 0x0080,  // 64-bit addresses allowed
    MsiValue_FlagsMaskBit    = 0x0100,  // Per-vector masking capable
};

/**
 * @brief advanced feature structure offsets
 */
enum AdvancedFeaturesOffset
{
    AdvancedFeaturesOffset_Length = 2,
    AdvancedFeaturesOffset_Cap    = 3,
    AdvancedFeaturesOffset_Ctrl   = 4,
    AdvancedFeaturesOffset_Status = 5,
};

 /**
  * @brief advanced feature configuration space values
  */
enum AdvancedFeaturesValue
{
    AdvancedFeaturesValue_CapTp    = 0x01,
    AdvancedFeaturesValue_CapFlr   = 0x02,
    AdvancedFeaturesValue_CtrlFlr  = 0x01,
    AdvancedFeaturesValue_StatusTp = 0x01,
};

/**
 * @brief Legacy PCI X, type 0 (non-bridge) devices
 *        configuration structure sizes
 */
const size_t PciXType0CapV0Size = 8;   // size of registers for Version 0
const size_t PciXType0CapV1Size = 24;  // size of registers for Version 1
const size_t PciXType0CapV2Size = 24;  // size of registers for Version 2

/**
 * @brief Legacy PCI X, type 0 (non-bridge) devices
 *        configuration space offsets
 */
enum PciXType0Offset
{
    PciXType0Offset_Command      = 2,   // Modes & Features
    PciXType0Offset_Status       = 4,   // PCI-X capabilities
    PciXType0Offset_StatusEccCsr = 8,   // ECC control and status
};

/**
 * @brief Legacy PCI X, type 0 (non-bridge) devices
 *        configuration space values
 */
enum PciXType0Value
{
    PciXType0Value_CommandDataParityErrorEnable  = 0x0001,      // Data Parity Error Recovery Enable
    PciXType0Value_CommandEnableRelaxedOrdering  = 0x0002,      // Enable Relaxed Ordering
    PciXType0Value_CommandRead512                = 0x0000,      // 512 byte maximum read byte count
    PciXType0Value_CommandRead1K                 = 0x0004,      // 1Kbyte maximum read byte count
    PciXType0Value_CommandRead2K                 = 0x0008,      // 2Kbyte maximum read byte count
    PciXType0Value_CommandRead4K                 = 0x000c,      // 4Kbyte maximum read byte count
    PciXType0Value_CommandMaxRead                = 0x000c,      // Max Memory Read Byte Count
    PciXType0Value_CommandSplit1                 = 0x0000,      // Max 1
    PciXType0Value_CommandSplit2                 = 0x0010,      // Max 2
    PciXType0Value_CommandSplit3                 = 0x0020,      // Max 3
    PciXType0Value_CommandSplit4                 = 0x0030,      // Max 4
    PciXType0Value_CommandSplit8                 = 0x0040,      // Max 8
    PciXType0Value_CommandSplit12                = 0x0050,      // Max 12
    PciXType0Value_CommandSplit16                = 0x0060,      // Max 16
    PciXType0Value_CommandSplit32                = 0x0070,      // Max 32
    PciXType0Value_CommandMaxSplit               = PciXType0Value_CommandSplit32,

    PciXType0Value_StatusDeviceFn                = 0x000000ff,  // A copy of devfn
    PciXType0Value_StatusBus                     = 0x0000ff00,  // A copy of bus nr
    PciXType0Value_Status64Bit                   = 0x00010000,  // 64-bit device
    PciXType0Value_Status133Mhz                  = 0x00020000,  // 133 MHz capable
    PciXType0Value_StatusSplDisc                 = 0x00040000,  // Split Completion Discarded
    PciXType0Value_StatusUnxSpl                  = 0x00080000,  // Unexpected Split Completion
    PciXType0Value_StatusComplex                 = 0x00100000,  // Device Complexity
    PciXType0Value_StatusMaxRead                 = 0x00600000,  // Designed Max Memory Read Count
    PciXType0Value_StatusMaxSplit                = 0x03800000,  // Designed Max Outstanding Split Transactions
    PciXType0Value_StatusMaxCum                  = 0x1c000000,  // Designed Max Cumulative Read Size
    PciXType0Value_StatusSplErr                  = 0x20000000,  // Rcvd Split Completion Error Msg
    PciXType0Value_Status266Mhz                  = 0x40000000,  // 266 MHz capable
    PciXType0Value_Status533Mhz                  = 0x80000000,  // 533 MHz capable
};

/**
 * @brief Legacy PCI X, PCI-to-PCI bridges of type '1'
 *        configuration space offsets
 */
enum PciXType1Offset
{
    PciXType1Offset_SecondaryStatus = 2,    // Secondary Status
    PciXType1Offset_Status          = 4,    // Bridge Status
};

/**
 * @brief Legacy PCI X, PCI-to-PCI bridges of type '1'
 *        configuration space values
 */
enum PciXType1Value
{
    PciXType1Value_SecondaryStatus64Bit  = 0x0001,  // Secondary AD interface is 64 bits
    PciXType1Value_SecondaryStatus133Mhz = 0x0002,  // 133 MHz capable
    PciXType1Value_SecondaryStatusFreq   = 0x03c0,  // Secondary Bus Mode and Frequency
    PciXType1Value_SecondaryStatusVers   = 0x3000,  // PCI-X Capability Version
    PciXType1Value_SecondaryStatusV1     = 0x1000,  // Mode 2, not Mode 1
    PciXType1Value_SecondaryStatusV2     = 0x2000,  // Mode 1 or Modes 1 and 2
    PciXType1Value_SecondaryStatus266Mhz = 0x4000,  // 266 MHz capable
    PciXType1Value_SecondaryStatus533Mhz = 0x8000,  // 533 MHz capable
};

/**
 * @brief PCI Express (PCIe) Register Offsets
 */
enum PcieOffset
{
    PcieOffset_Flags                 = 2,
    PcieOffset_DeviceCapability      = 4,
    PcieOffset_DeviceControl         = 8,
    PcieOffset_DeviceStatus          = 10,
    PcieOffset_LinkCapability        = 12,
    PcieOffset_LinkControl           = 16,
    PcieOffset_LinkStatus            = 18,
    PcieOffset_SlotCapability        = 20,
    PcieOffset_SlotControl           = 24,
    PcieOffset_SlotStatus            = 26,
    PcieOffset_RootControl           = 28,
    PcieOffset_RootCapabilities      = 30,
    PcieOffset_RootStatus            = 32,
    PcieOffset_DeviceCapability2     = 36,
    PcieOffset_DeviceControl2        = 40,
    PcieOffset_DeviceStatus2         = 42,
    PcieOffset_LinkCapability2       = 44,
    PcieOffset_LinkControl2          = 48,
    PcieOffset_LinkStatus2           = 50,
    PcieOffset_SlotCapability2       = 52,
    PcieOffset_SlotControl2          = 56,
    PcieOffset_SlotStatus2           = 58,
};

/**
 * @brief PCI Express (PCIe) Register Values
 */
enum PcieValue
{
    PcieValue_FlagsVersion                                      = 0x000f,       // Capability version
    PcieValue_FlagsType                                         = 0x00f0,       // Device/Port type
    PcieValue_TypeEndpoint                                      = 0x0,          // Express Endpoint
    PcieValue_TypeLegacyEndpoint                                = 0x1,
    PcieValue_TypeRootPort                                      = 0x4,
    PcieValue_TypeUpstreamPort                                  = 0x5,
    PcieValue_TypeDownstreamPort                                = 0x6,
    PcieValue_TypePciBridge                                     = 0x7,          // PCIe to PCI/PCI-X Bridge
    PcieValue_TypePcieBridge                                    = 0x8,          // PCI/PCI-X to PCIe Bridge
    PcieValue_TypeRootComplexEndpoint                           = 0x9,
    PcieValue_TypeRootComplexEventCollector                     = 0xa,
    PcieValue_FlagsSlot                                         = 0x0100,
    PcieValue_FlagsIrq                                          = 0x3e00,

    PcieValue_DeviceCapabilityPayload                           = 0x00000007,
    PcieValue_DeviceCapabilityPhantomFunctions                  = 0x00000018,
    PcieValue_DeviceCapabilityExtendedTags                      = 0x00000020,
    PcieValue_DeviceCapabilityL0sAcceptableLatency              = 0x000001c0,
    PcieValue_DeviceCapabilityL0sAcceptableLatency_Shift        = 6,
    PcieValue_DeviceCapabilityL1AcceptableLatency               = 0x00000e00,
    PcieValue_DeviceCapabilityL1AcceptableLatency_Shift         = 9,
    PcieValue_DeviceCapabilityAttentionButtonPresent            = 0x00001000,
    PcieValue_DeviceCapabilityAttentionButtonIndicatorPresent   = 0x00002000,
    PcieValue_DeviceCapabilityPowerIndicatorPresent             = 0x00004000,
    PcieValue_DeviceCapabilityRoleBasedErrorReporting           = 0x00008000,
    PcieValue_DeviceCapabilityPowerLimitValue                   = 0x03fc0000,
    PcieValue_DeviceCapabilitySlotPowerLimitScale               = 0x0c000000,
    PcieValue_DeviceCapabilityFunctionLevelReset                = 0x10000000,

    PcieValue_DeviceControlCorrectableErrorReportingEnable      = 0x0001,
    PcieValue_DeviceControlNonFatalErrorReportingEnable         = 0x0002,
    PcieValue_DeviceControlFatalErrorReportingEnable            = 0x0004,
    PcieValue_DeviceControlUunsupportedRequestReportingEnable   = 0x0008,
    PcieValue_DeviceControlRelaxedOrderingEnable                = 0x0010,
    PcieValue_DeviceControlPayload                              = 0x00e0,
    PcieValue_DeviceControlExtendedTagFieldEnable               = 0x0100,
    PcieValue_DeviceControlPhantomFunctionEnable                = 0x0200,
    PcieValue_DeviceControlAuxPowerManagementEnable             = 0x0400,
    PcieValue_DeviceControlNoSnoopEnable                        = 0x0800,
    PcieValue_DeviceControlMaxReadRequestSize                   = 0x7000,
    PcieValue_DeviceControlBridgeConfigurationRetry             = 0x8000,
    PcieValue_DeviceControlFunctionLevelReset                   = 0x8000,

    PcieValue_DeviceStatusCorrectableErrorDetected              = 0x0001,
    PcieValue_DeviceStatusNonFatalErrorDetected                 = 0x0002,
    PcieValue_DeviceStatusFatalErrorDetected                    = 0x0004,
    PcieValue_DeviceStatusRequestDetected                       = 0x0008,
    PcieValue_DeviceStatusAuxPowerDetected                      = 0x0010,
    PcieValue_DeviceStatusTransactionPending                    = 0x0020,

    PcieValue_LinkCapabilitySupportedLinkSpeeds                 = 0x0000000f,
    PcieValue_LinkCapabilitySupportedLinkSpeed_2_5GB            = 0x00000001,
    PcieValue_LinkCapabilitySupportedLinkSpeed_5_0GB            = 0x00000002,
    PcieValue_LinkCapabilityMaximumLinkWidth                    = 0x000003f0,
    PcieValue_LinkCapabilityAspmSupport                         = 0x00000c00,
    PcieValue_LinkCapabilityAspmSupport_Shift                   = 10,
    PcieValue_LinkCapabilityL0sExitLatency                      = 0x00007000,
    PcieValue_LinkCapabilityL0sExitLatency_Shift                = 12,
    PcieValue_LinkCapabilityL1ExitLatency                       = 0x00038000,
    PcieValue_LinkCapabilityL1ExitLatency_Shift                 = 15,
    PcieValue_LinkCapabilityClockPowerManagement                = 0x00040000,
    PcieValue_LinkCapabilitySurproseDownErrorReportingCapable   = 0x00080000,
    PcieValue_LinkCapabilityDataLinkLayerActiveReportingCapable = 0x00100000,
    PcieValue_LinkCapabilityLinkBandwidthNotificationCapability = 0x00200000,
    PcieValue_LinkCapabilityPortNumber                          = 0xff000000,

    PcieValue_LinkControlAspmControl                            = 0x0003,
    PcieValue_LinkControlAspmL0sEnable                          = 0x0001,
    PcieValue_LinkControlAspmL1Enable                           = 0x0002,
    PcieValue_LinkControlReadCompletionBoundary                 = 0x0008,
    PcieValue_LinkControlLinkDisable                            = 0x0010,
    PcieValue_LinkControlRetrainLink                            = 0x0020,
    PcieValue_LinkControlCommonClockConfiguration               = 0x0040,
    PcieValue_LinkControlExtebdedSync                           = 0x0080,
    PcieValue_LinkControlEnableClockReq                         = 0x0100,
    PcieValue_LinkControlHardwareAutonomousWidthDisable         = 0x0200,
    PcieValue_LinkControlLinkBandwidthManamgentInterruptEnable  = 0x0400,
    PcieValue_LinkControlLinkAutonomousBandwidthInterruptEnable = 0x0800,

    PcieValue_LinkStatusCurrentLinkSpeed                        = 0x000f,
    PcieValue_LinkStatusCurrentLinkSpeed_2_5GB                  = 0x0001,
    PcieValue_LinkStatusCurrentLinkSpeed_5_0GB                  = 0x0002,
    PcieValue_LinkStatusCurrentLinkSpeed_8_0GB                  = 0x0003,
    PcieValue_LinkStatusNegotiatedLinkWidth                     = 0x03f0,
    PcieValue_LinkStatusNegotiatedLinkWidthX1                   = 0x0010,
    PcieValue_LinkStatusNegotiatedLinkWidthX2                   = 0x0020,
    PcieValue_LinkStatusNegotiatedLinkWidthX4                   = 0x0040,
    PcieValue_LinkStatusNegotiatedLinkWidthX8                   = 0x0080,
    PcieValue_LinkStatusNegotiatedLinkWidthShift                = 4,
    PcieValue_LinkStatusLinkTraining                            = 0x0800,
    PcieValue_LinkStatusSlotClockConfiguration                  = 0x1000,
    PcieValue_LinkStatusDataLinkLayerActive                     = 0x2000,
    PcieValue_LinkStatusLinkBandwidthManagementStatus           = 0x4000,
    PcieValue_LinkStatusLinkAutonomousBandwidthStatus           = 0x8000,

    PcieValue_SlotCapabilityAttentionButtonPresent              = 0x00000001,
    PcieValue_SlotCapabilityPowerControllerPresent              = 0x00000002,
    PcieValue_SlotCapabilityMrlSensorPresent                    = 0x00000004,
    PcieValue_SlotCapabilityAttentionIndicatorPresent           = 0x00000008,
    PcieValue_SlotCapabilityPowerIndicatorPresent               = 0x00000010,
    PcieValue_SlotCapabilityHotPlugSurpose                      = 0x00000020,
    PcieValue_SlotCapabilityHotPlugCapable                      = 0x00000040,
    PcieValue_SlotCapabilitySlotPowerLimitValue                 = 0x00007f80,
    PcieValue_SlotCapabilitySlotPowerLimitScale                 = 0x00018000,
    PcieValue_SlotCapabilityElectromechanicamInterlockPresent   = 0x00020000,
    PcieValue_SlotCapabilityNoCommandCompletedSupport           = 0x00040000,
    PcieValue_SlotCapabilityPhysicalSlotNumber                  = 0xfff80000,

    PcieValue_SlotControlAttentionButtonPressedEnable           = 0x0001,
    PcieValue_SlotControlPowerFaultDetectedEnable               = 0x0002,
    PcieValue_SlotControlMrlSensorChangedEnable                 = 0x0004,
    PcieValue_SlotControlPresenceDetectChangedEnable            = 0x0008,
    PcieValue_SlotControlCommandCompletedInterruptEnable        = 0x0010,
    PcieValue_SlotControlHotPlugInterruptEnable                 = 0x0020,
    PcieValue_SlotControlAttentionIndicatorControl              = 0x00c0,
    PcieValue_SlotControlAttentionIndicatorOn                   = 0x0040,
    PcieValue_SlotControlAttentionIndicatorBlinking             = 0x0080,
    PcieValue_SlotControlAttentionIndicatorOff                  = 0x00c0,
    PcieValue_SlotControlPowerIndicatorControl                  = 0x0300,
    PcieValue_SlotControlPowerIndicatorOn                       = 0x0100,
    PcieValue_SlotControlPowerIndicatorBlinking                 = 0x0200,
    PcieValue_SlotControlPowerIndicatorOff                      = 0x0300,
    PcieValue_SlotControlPowerControllerControl                 = 0x0400,
    PcieValue_SlotControlPowerOn                                = 0x0000,
    PcieValue_SlotControlPowerOff                               = 0x0400,
    PcieValue_SlotControlElectromechanicalInterlockControl      = 0x0800,
    PcieValue_SlotControlDataLinkLayerStateChangedEnable        = 0x1000,

    PcieValue_SlotStatusAttentionButtonPressed                  = 0x0001,
    PcieValue_SlotStatusPowerFaultDetected                      = 0x0002,
    PcieValue_SlotStatusMrlSensorChanged                        = 0x0004,
    PcieValue_SlotStatusPresenceDetectChanged                   = 0x0008,
    PcieValue_SlotStatusCommandCompleted                        = 0x0010,
    PcieValue_SlotStatusMrlSensorState                          = 0x0020,
    PcieValue_SlotStatusPresenceDetectState                     = 0x0040,
    PcieValue_SlotStatusElectromechanicalInterlockStatus        = 0x0080,
    PcieValue_SlotStatusDataLinkLayerStateChanged               = 0x0100,

    PcieValue_RootControlSystemErrorCorrectableError            = 0x0001,
    PcieValue_RootControlSystemErrorNonFatalError               = 0x0002,
    PcieValue_RootControlSystemErrorFatalError                  = 0x0004,
    PcieValue_RootControlPmeInterruptEnable                     = 0x0008,
    PcieValue_RootControlCrsSoftwareVisibilityEnable            = 0x0010,

    PcieValue_RootCapabilityCrsSoftwareVisibility               = 0x0001,

    PcieValue_RootStatusPmeStatus                               = 0x00010000,
    PcieValue_RootStatusPmePending                              = 0x00020000,

    // The following are present only on "PCIe Capability version 2" devices
    PcieValue_DeviceCapability2AlternativeRoutingId             = 0x00000020,
    PcieValue_DeviceCapability2LatencyToleranceReporting        = 0x00000800,
    PcieValue_DeviceCapability2ObffSupoport                     = 0x000c0000,
    PcieValue_DeviceCapability2ObffMessage                      = 0x00040000,
    PcieValue_DeviceCapability2ObffWake                         = 0x00080000,

    PcieValue_DeviceControl2CompletionTimeoutValue              = 0x000f,
    PcieValue_DeviceControl2AlternativeRoutingId                = 0x0020,
    PcieValue_DeviceControl2IdoRequestEnable                    = 0x0100,
    PcieValue_DeviceControl2IdoCompletionEnable                 = 0x0200,
    PcieValue_DeviceControl2LtrEnable                           = 0x0400,
    PcieValue_DeviceControl2ObffMessageAEnable                  = 0x2000,
    PcieValue_DeviceControl2ObffMessageBEnable                  = 0x4000,
    PcieValue_DeviceControl2ObffWakeEnable                      = 0x6000,

    PcieValue_LinkCapability2SupportedLinkSpeed_2_5GB           = 0x00000002,
    PcieValue_LinkCapability2SupportedLinkSpeed_5_0GB           = 0x00000004,
    PcieValue_LinkCapability2SupportedLinkSpeed_8_0GB           = 0x00000008,
    PcieValue_LinkCapability2CrosslinkSupported                 = 0x00000100,
};

/**
 * @brief Extended capability IDs (PCI-X 2.0 and beyond)
 */
enum ExtendedCapabilityId
{
    ExtendedCapabilityId_AdvancedErrorReporting           = 0x01,
    ExtendedCapabilityId_VirtualChannelCapability         = 0x02,
    ExtendedCapabilityId_DeviceSerialNumber               = 0x03,
    ExtendedCapabilityId_PowerBudgeting                   = 0x04,
    ExtendedCapabilityId_RootComplexLinkDeclaration       = 0x05,
    ExtendedCapabilityId_RootComplexInternalLinkControl   = 0x06,
    ExtendedCapabilityId_RootComplexEventCollector        = 0x07,
    ExtendedCapabilityId_MultiFunctionVcCapability        = 0x08,
    ExtendedCapabilityId_Vc9                              = 0x09,
    ExtendedCapabilityId_RootComplexRb                    = 0x0A,
    ExtendedCapabilityId_VendorSpecific                   = 0x0B,
    ExtendedCapabilityId_AccessControlServices            = 0x0D,
    ExtendedCapabilityId_AlternateRoutingId               = 0x0E,
    ExtendedCapabilityId_AddressTranslationServices       = 0x0F,
    ExtendedCapabilityId_SingleRootIoVirtualization       = 0x10,
    ExtendedCapabilityId_MultiRootIoVirtualization        = 0x11,
    ExtendedCapabilityId_Multicast                        = 0x12,
    ExtendedCapabilityId_PageRequestInterface             = 0x13,
    ExtendedCapabilityId_ResizableBAR                     = 0x15,
    ExtendedCapabilityId_DynamicPowerAllocation           = 0x16,
    ExtendedCapabilityId_TPHRequester                     = 0x17,
    ExtendedCapabilityId_LatencyToleranceReporting        = 0x18,
    ExtendedCapabilityId_SecondaryPCIeCapability          = 0x19,
    ExtendedCapabilityId_ProtocolMultiplexing             = 0x1A,
    ExtendedCapabilityId_ProcessAddressSpaceId            = 0x1B,
    ExtendedCapabilityId_L1Substates                      = 0x1E,  // L1SS
    ExtendedCapabilityId_Max                              = ExtendedCapabilityId_L1Substates
};

/**
 * @brief Expected size of DeviceSerialNumber ExtendedCapability
 */
const size_t ExtendedCapabilityDeviceSerialNumberSize = 12;

/**
 * @brief Extended capability offsets
 */
enum ExtendedCapabilityOffset {
    ExtendedCapabilityOffset_L1SSCap           = 0x4,       // L1SS_PM_SS_EXT_CAP_OFFSET
    ExtendedCapabilityOffset_L1SSCtrl1         = 0x8,       // L1SS_PM_SS_EXT_CTRL1_OFFSET
    ExtendedCapabilityOffset_L1SSCtrl2         = 0xc        // L1SS_PM_SS_EXT_CTRL2_OFFSET
};

/**
 * @brief Extended capability values
 */
enum ExtendedCapabilityValue {
    ExtendedCapabilityValue_L1SSCap                     = 1 << 4,    // L1SS_CAP_L1SS
    ExtendedCapabilityValue_L1SSCapL11ASPM              = 1 << 3,    // L1SS_CAP_L11_ASPM
    ExtendedCapabilityValue_L1SSCapL12ASPM              = 1 << 2,    // L1SS_CAP_L12_ASPM
    ExtendedCapabilityValue_L1SSCapL11PM                = 1 << 1,    // L1SS_CAP_L11_PM
    ExtendedCapabilityValue_L1SSCapL12PM                = 1 << 0,    // L1SS_CAP_L12_PM
    ExtendedCapabilityValue_L1SSCapTPowerOnScaleShift   = 16,        // L1SS_CAP_TPOWERON_SCALE_SHIFT
    ExtendedCapabilityValue_L1SSCapTPowerOnValueShift   = 19,        // L1SS_CAP_TPOWERON_VALUE_SHIFT
    ExtendedCapabilityValue_L1SSTPowerOnScaleMask       = 0x3,       // L1SS_TPOWERON_SCALE_MASK
    ExtendedCapabilityValue_L1SSTPowerOnValueMask       = 0xf,       // L1SS_TPOWERON_VALUE_MASK
    ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleShift = 29,        // L1SS_CTRL1_L12_THRES_SCALE_SHIFT
    ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleMask  = 0x7,       // L1SS_CTRL1_L12_THRES_SCALE_MASK
    ExtendedCapabilityValue_L1SSCtrl1L12ThresValueShift = 16,        // L1SS_CTRL1_L12_THRES_VALUE_SHIFT
    ExtendedCapabilityValue_L1SSCtrl1L12ThresValueMask  = 0x3ff,     // L1SS_CTRL1_L12_THRES_VALUE_MASK
    ExtendedCapabilityValue_L1SSCtrl2TPowerOnScaleShift = 0,         // L1SS_CTRL2_TPOWERON_SCALE_SHIFT
    ExtendedCapabilityValue_L1SSCtrl2TPowerOnValueShift = 3,         // L1SS_CTRL2_TPOWERON_VALUE_SHIFT
    ExtendedCapabilityValue_L1SSCtrl2TPowerOnVSMask     = (ExtendedCapabilityValue_L1SSTPowerOnScaleMask << ExtendedCapabilityValue_L1SSCtrl2TPowerOnScaleShift) | (ExtendedCapabilityValue_L1SSTPowerOnValueMask << ExtendedCapabilityValue_L1SSCtrl2TPowerOnValueShift),
    ExtendedCapabilityValue_L1SSPmSsCmRsTimeMask        = 0xff,      //
    ExtendedCapabilityValue_L1SSPmSsCmRsTimeShift       = 8          // L1SS_PM_SS_CM_RSTIME_SHIFT
};

} // namespace pcie
} // namespace nn


