﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief PCIe Driver LoggedState
 *
 * @details This file defines the API and data types for the LoggedState feature.
 *          This file should be used only for telemetry and diagnostic purposes.
 */

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn {  namespace pcie {

/**
 * @brief Root port state flags
 */
struct RootPortStateFlag
{
    using IsEnabled               = nn::util::BitFlagSet<32, RootPortStateFlag>::Flag<0> ; // <! Is port enabled?
    using DataLinkActiveStandard  = nn::util::BitFlagSet<32, RootPortStateFlag>::Flag<1> ; // <! Is link active?
    using DataLinkActiveVendor    = nn::util::BitFlagSet<32, RootPortStateFlag>::Flag<2> ; // <! Is link active?
};
using RootPortStateFlagSet = nn::util::BitFlagSet<32, RootPortStateFlag>;

/**
 * @brief Root Port log information
 */
struct RootPortLoggedState
{
    BusSpeed             speed;                // <! Negotiated PCIe link speed
    RootPortStateFlagSet flagSet;              // <! Bitmapped set of flags to express state of root port
    uint32_t             portResetTimeInUs;    // <! Upon enumeration, how long in us did link take to come up?
    uint32_t             irqCount;             // <! IRQ count

    // per-port statistics
    uint32_t afiMsgPmeCount;                   // <! Power management event count
    uint32_t afiMsgIntxCount;                  // <! Legacy INTX count
    uint32_t afiMsgPcieFatalErrorCount;        // <! Fatal error message received from endpoint count
    uint32_t afiMsgPcieNonFatalErrorCount;     // <! Non-fatal error message received from endpoint count
    uint32_t afiMsgPcieCorrectableErrorCount;  // <! Correctable error message received from endpoint count
    uint32_t afiMsgPcieRootPortIntCount;       // <! Endpoint hotplug count
    uint32_t afiMsgHotplugCount;
};

const RootPortLoggedState InvalidRootPortLoggedState = {BusSpeed_Invalid, {{0}}, 0, 0, 0, 0, 0, 0, 0, 0, 0};

/**
 * @brief Root Complex global count, all ports scope
 */
struct RootComplexGlobalCounts
{
    uint32_t afiInitiatorSlaveReadErrorCount;   // <! AXI slave read error count
    uint32_t afiInitiatorSlaveWriteErrorCount;  // <! AXI slave write error count
    uint32_t afiInitiatorDecodeReadErrorCount;  // <! AXI decode read error count
    uint32_t afiInitiatorDecodeWriteErrorCount; // <! AXI decode write error count
    uint32_t afiTargetSlaveReadErrorCount;      // <! Target read abort error count
    uint32_t afiTargetSlaveWriteErrorCount;     // <! Target write abort error count
    uint32_t afiTargetDecodeReadErrorCount;     // <! Master read abort error count
    uint32_t afiTargetDecodeWriteErrorCount;    // <! Master write abort error count
    uint32_t afiTargetWriteErrorCount;          // <! Invalid write error count
    uint32_t afiDfpciDecodeReadErrorCount;      // <! Read response decoding error count
    uint32_t afiDfpciDecodeWriteErrorCount;     // <! Write response decoding error count
    uint32_t afiFpciReadTimeoutCount;           // <! Read transaction timeout count
    uint32_t afiFpciWriteTimeoutCount;          // <! Write transaction timeout count
    uint32_t afiPePresentSenseCount;            // <! Present sense change count
    uint32_t afiPeClockReqSenseCount;           // <! Slot clock request change count
    uint32_t afiClockClampSenseCount;           // <! TMS clock clamp change count
    uint32_t afiReadyForPowerDownSenseCount;    // <! TMS power down count
    uint32_t afiPeerToPeerErrorCount;           // <! Peer to peer error count
    uint32_t intVectorWakeCount;                // <! Wake interrupt vector count
    uint32_t intVectorMsiCount;                 // <! MSI interrupt vector count
    uint32_t intVectorIntCount;                 // <! INT interrupt vector count
};

const RootComplexGlobalCounts InvalidRootComplexGlobalCounts = {};

/**
 * @brief Root Complex log information
 */
struct RootComplexLoggedState
{
    // statistics with global (all ports) scope
    RootComplexGlobalCounts globalCounts;

    // resistor calibration
    uint8_t pllResistorCalibrationValue;

    // per-port state and statistics
    RootPortLoggedState ports[MaxSwitchPortCount];
};

const RootComplexLoggedState InvalidRootComplexLoggedState = {InvalidRootComplexGlobalCounts, 0,
    {InvalidRootPortLoggedState,InvalidRootPortLoggedState}};



/**
 * @brief Summary of log information
 */
struct LoggedState
{
    RootComplexLoggedState  rootComplex;
    FunctionState           functions[MaxSwitchPortCount];
};

const LoggedState InvalidLoggedState = {InvalidRootComplexLoggedState, {InvalidFunctionState, InvalidFunctionState}};

/**
 * @brief Initialize Logged State PCIe API
 * @retresult
 *     @handleresult{ResultSuccess,           Initialization succeeded}
 * @endretresult
 * @pre
 *   - Initialize() has not yet been called, or
 *   - Finalize() has been previously called.
 * @post
 *   Logged State PCIe API can now be used.
 * @details
 *   This method prepares API for use, and thus must be called first.
 */
Result InitializeLoggedState() NN_NOEXCEPT;

/**
 * @brief Finalize Logged State PCIe API
 * @retresult
 *     @handleresult{ResultSuccess,           Initialization succeeded}
 * @endretresult
 * @pre
 *   - InitializeLoggedState() has been previously called.
 * @post
 *   - Logged State PCIe API can no longer be used.
 * @details
 *   This method releases internal resources, after which no other API may be called.
 */
Result FinalizeLoggedState() NN_NOEXCEPT;

/**
 * @brief  Retrieve logged state.
 * @param[out] pOutput
 *   Output logged state data
 * @param[out] pOutIsLoggedErrorPending
 *   Output logged error pending flag
 * @param[in]  isCleared
 *   'true' to request that cumulative state be cleared after retrieval.
 *   'false' to allow cumulative state to persist.
 * @retresult
 *     @handleresult{ResultNoController,             Root Complex hardware not available at this time}
 * @endretresult
 * @pre
 *   Initialize() must be called once beforehand.
 * @post
 *   Cumulative state is cleared if isCleared was specified as true.
 * @details
 *   Class drivers should not use this API. It is intended for use by telemetry.
 */
Result GetLoggedState(LoggedState* pOutput, bool* pOutIsLoggedErrorPending, bool isCleared) NN_NOEXCEPT;

/**
 * @brief  Print logged state.
 * @param[in] loggedState
 *   Logged state to be printed
 * @param[in] printfCallback
 *   Callback invoke for each resulting formatted output
 * @pre
 *   GetLoggedState() was called to retrieve logged state structure.
 * @post
 *   Does not affect any state, only prints.
 * @details
 *   Class drivers should not use this API. It is intended for use by telemetry.
 */
void PrintLoggedState(nn::pcie::LoggedState& loggedState, void (*printfCallback) (const char *str, ...)) NN_NOEXCEPT;

/**
 * @brief  Get logged state event.
 * @param[out] pOutLoggedStateEvent
 *   Supplied system event is initialized, and will provide indication of when logged state is updated
 * @param[in] eventClearMode
 *   Mode of operation assumed by initialized system event, related
 *   to how set event is cleared.
 * @param[in]  isStatisticReported
 *   'true' event will signal even when benign statistics are updated.
 *   'false' event will signal only upon accumulation of errors.
 * @pre
 *   Initialize() must be called once beforehand.
 * @post
 *   Supplied event is initialized.
 * @details
 *   Class drivers should not use this API. It is intended for use by telemetry.
 */
Result GetLoggedStateEvent(nn::os::SystemEventType* pOutLoggedStateEvent,
                           nn::os::EventClearMode eventClearMode,
                           bool isStatisticReported) NN_NOEXCEPT;


} // namespace pcie
} // namespace nn
