﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/olsc/srv/database/olsc_CachedReaderWriter.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <utility>

namespace nn { namespace olsc { namespace srv { namespace database {

template<typename KeyType, typename ValueType, int ReadBufferCount>
class SortedDataArray
{
    NN_DISALLOW_COPY(SortedDataArray);
    NN_DISALLOW_MOVE(SortedDataArray);

public:
    struct KeyValueType
    {
        KeyType     key;
        ValueType   value;
    };

    using Compare = std::function<int(const KeyType& lhs, const KeyType& rhs)>;
    using ReadBuffer = std::array<KeyValueType, ReadBufferCount>;

    SortedDataArray(const Compare& compare, ReadBuffer* readBuffer, int maxEntryCount) NN_NOEXCEPT :
        m_ReaderWriter(readBuffer->data(), static_cast<int>(readBuffer->size()), maxEntryCount), m_MaxEntryCount(maxEntryCount), m_Compare(compare)
    {}

    // TODO: 現状 Add 内で Update 的なこともやっているが動作が不明確なので分けたい。
    //       Add: 同一の key が存在していたら失敗
    //       Update: 同一の key が存在しないなら失敗
    bool Add(const KeyType& key, const ValueType& value) NN_NOEXCEPT;
    void Remove(const KeyType& key) NN_NOEXCEPT;
    nn::util::optional<ValueType> Find(const KeyType& key) const NN_NOEXCEPT;
    const KeyValueType operator[](int index) const NN_NOEXCEPT;

    int GetCount() const NN_NOEXCEPT;
    void Cleanup() NN_NOEXCEPT;

    // std::lock_guard 用
    void lock() NN_NOEXCEPT;
    void lock() const NN_NOEXCEPT;
    void unlock() NN_NOEXCEPT;
    void unlock() const NN_NOEXCEPT;

protected:
    Result InitializeUnsafe() NN_NOEXCEPT;
    virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT = 0;
    virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT = 0;
    virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT = 0;
    virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT = 0;

private:
    const char* MakeMetadataFilePath(char* out, int maxOutLen, const char* rootPath) const NN_NOEXCEPT;
    const char* MakeEntryFilePath(char* out, int maxOutLen, const char* rootPath) const NN_NOEXCEPT;
    Result ReadMetaData() NN_NOEXCEPT;
    Result WriteMetaData() NN_NOEXCEPT;

    std::pair<int, bool> FindIndex(const char* path, const KeyType& key) const NN_NOEXCEPT;
    void GetKeyValue(KeyValueType* out, const char* path, int index) const NN_NOEXCEPT;

    static const int MaxPathLen = 64;

    mutable CachedReaderWriter<KeyValueType> m_ReaderWriter;

    struct Metadata
    {
        int entryCount;
    };
    Metadata m_Metadata = {};

    int m_MaxEntryCount;
    mutable os::SdkRecursiveMutex m_Lock;

    Compare m_Compare;
};


}}}} // namespace nn::olsc::srv::database

#define NN_OLSC_DEFINE_SORTED_DATA_ARRAY_TYPE \
SortedDataArray<KeyType, ValueType, ReadBufferCount>

#define NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(rtype, methodInfo) \
template <typename KeyType, typename ValueType, int ReadBufferCount> \
inline rtype NN_OLSC_DEFINE_SORTED_DATA_ARRAY_TYPE::methodInfo NN_NOEXCEPT

namespace nn { namespace olsc { namespace srv { namespace database {

// -------------------------------------------------------------------------------------------------
// Private
// -------------------------------------------------------------------------------------------------

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(const char*, MakeMetadataFilePath(char* out, int maxOutLen, const char* rootPath) const)
{
    NN_ABORT_UNLESS(nn::util::TSNPrintf(out, maxOutLen, "%s%s", rootPath, GetMetadataFileRelativePath()) < maxOutLen);
    return out;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(const char*, MakeEntryFilePath(char* out, int maxOutLen, const char* rootPath) const)
{
    NN_ABORT_UNLESS(nn::util::TSNPrintf(out, maxOutLen, "%s%s", rootPath, GetEntryFileRelativePath()) < maxOutLen);
    return out;
}

#define INT_BOOL_PAIR std::pair<int, bool>
NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(
    INT_BOOL_PAIR, FindIndex(const char* path, const KeyType& key) const)
{
    auto low = 0;
    auto high = m_Metadata.entryCount;

    while (low != high)
    {
        auto index = (low + high) / 2;
        KeyValueType kv;
        GetKeyValue(&kv, path, index);
        auto compResult = m_Compare(key, kv.key);
        if (compResult == 0)
        {
            return std::pair<int, bool>(index, true);
        }
        else if (compResult < 0) // key < keyValue.key
        {
            high = index;
        }
        else // key >= keyValue.key
        {
            low = (low == index) ? index + 1 : index;
        }
    }

    NN_SDK_ASSERT(low >= 0 && low <= m_Metadata.entryCount);

    // low が末尾要素の次を指す場合は KeyValue の取得・比較が不可能なので match = false として返る
    // これは key がどの要素よりも大きいケース
    if (low == m_Metadata.entryCount)
    {
        return std::pair<int, bool>(low, false);
    }

    KeyValueType kv;
    GetKeyValue(&kv, path, low);

    return std::pair<int, bool>(low, m_Compare(key, kv.key) == 0);
}
#undef INT_BOOL_PAIR


NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, GetKeyValue(KeyValueType* out, const char* path, int index) const)
{
    NN_SDK_ASSERT(index >= 0 && index < m_MaxEntryCount);
    int readCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Read(&readCount, out, path, index, 1));
    NN_ABORT_UNLESS(readCount == 1);
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(Result, WriteMetaData())
{
    auto writeMount = AcquireWriteMount();

    char path[MaxPathLen];
    NN_RESULT_DO(util::WriteFile(MakeMetadataFilePath(path, MaxPathLen, writeMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata)));

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(Result, ReadMetaData())
{
    auto readMount = AcquireReadMount();

    char path[MaxPathLen];

    NN_RESULT_TRY(util::ReadFile(MakeMetadataFilePath(path, MaxPathLen, readMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata), 0))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            m_Metadata = {};
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(Result, InitializeUnsafe())
{
    NN_RESULT_DO(ReadMetaData());
    NN_RESULT_SUCCESS;
}

// -------------------------------------------------------------------------------------------------
// Public
// -------------------------------------------------------------------------------------------------

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(nn::util::optional<ValueType>, Find(const KeyType& key) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());

    auto readMount = AcquireReadMount();

    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, readMount.GetRootPath());

    auto indexAndMatch = FindIndex(path, key);
    if (indexAndMatch.second)
    {
        KeyValueType kv;
        GetKeyValue(&kv, path, indexAndMatch.first);
        return kv.value;
    }
    return nn::util::nullopt;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, Remove(const KeyType& key))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());

    auto writeMount = AcquireWriteMount();

    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath());

    auto indexAndMatch = FindIndex(path, key);
    if (!indexAndMatch.second)
    {
        return;
    }

    if (indexAndMatch.first < m_Metadata.entryCount - 1)
    {
        auto fromIndex = indexAndMatch.first + 1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Move(path, indexAndMatch.first, fromIndex, m_Metadata.entryCount - fromIndex));
    }

    m_Metadata.entryCount--;
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteMetaData());
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(bool, Add(const KeyType& key, const ValueType& value))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());

    auto writeMount = AcquireWriteMount();

    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath());

    KeyValueType newKeyValue = {
        key, value
    };

    auto indexAndMatch = FindIndex(path, key);
    if (indexAndMatch.second)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Write(path, &newKeyValue, indexAndMatch.first, 1));
        return true;
    }

    auto insertIndex = indexAndMatch.first;
    NN_SDK_ASSERT(m_Metadata.entryCount < m_MaxEntryCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Move(path, insertIndex + 1, insertIndex, m_Metadata.entryCount - insertIndex));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Write(path, &newKeyValue, insertIndex, 1));

    m_Metadata.entryCount++;
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteMetaData());
    return true;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, Cleanup())
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());

    auto writeMount = AcquireWriteMount();
    char path[MaxPathLen];
    NN_ABORT_UNLESS_RESULT_SUCCESS(util::DeleteFile(MakeMetadataFilePath(path, MaxPathLen, writeMount.GetRootPath())));
    NN_ABORT_UNLESS_RESULT_SUCCESS(util::DeleteFile(MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath())));
    m_ReaderWriter.Clear();
    m_Metadata = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteMetaData());
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(const typename NN_OLSC_DEFINE_SORTED_DATA_ARRAY_TYPE::KeyValueType, operator[](int index) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(index >= 0 && index < m_Metadata.entryCount);

    auto readMount = AcquireReadMount();

    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, readMount.GetRootPath());
    KeyValueType kv;
    GetKeyValue(&kv, path, index);
    return kv;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(int, GetCount() const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    return m_Metadata.entryCount;
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, lock())
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, lock() const)
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, unlock())
{
    m_Lock.Unlock();
}

NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD(void, unlock() const)
{
    m_Lock.Unlock();
}

}}}} // namespace nn::olsc::srv::database

#undef NN_OLSC_DEFINE_SORTED_DATA_ARRAY_TYPE
#undef NN_OLSC_DEFINE_SORTED_DATA_CONTAINER_METHOD
