﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/fs/fs_Result.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

template<typename EntryType>
class CachedReaderWriter
{
public:
    CachedReaderWriter(EntryType cacheBuffer[], int cacheBufferCount, int maxEntryCountInFile) NN_NOEXCEPT :
        m_CacheBuffer(cacheBuffer), m_CacheBufferCount(cacheBufferCount),
        m_CachedCount(0), m_CachedOffset(0), m_MaxEntryCountInFile(maxEntryCountInFile)
    {}

    Result Read(int* outCount, EntryType out[], const char* path, int offset, int count) NN_NOEXCEPT;
    Result Write(const char* path, const EntryType data[], int offset, int count) NN_NOEXCEPT;
    Result Move(const char* path, int toOffset, int fromOffset, int count) NN_NOEXCEPT;
    void Clear() NN_NOEXCEPT;

private:
    Result ReadImpl(int* outReadCount, EntryType out[], const char* path, int offset, int count) NN_NOEXCEPT;
    Result WriteImpl(const char* path, const EntryType data[], int offset, int count) NN_NOEXCEPT;
    Result ReadToCache(const char* path, int offset, int count) NN_NOEXCEPT;
    Result WriteFromCache(const char* path, int offset, int count) NN_NOEXCEPT;


    Result MoveToFront(const char* path, int toOffset, int fromOffset, int count) NN_NOEXCEPT;
    Result MoveToBack(const char* path, int toOffset, int fromOffset, int count) NN_NOEXCEPT;

    EntryType* m_CacheBuffer;
    int m_CacheBufferCount;
    int m_CachedCount;
    int m_CachedOffset;
    int m_MaxEntryCountInFile;
    os::SdkRecursiveMutex m_CacheBufferLock;
};



}}}} // namespace nn::olsc::srv::database

// ---------------------------------------------------------------------------------
// 実装
// ---------------------------------------------------------------------------------

#include <algorithm>
#include <mutex>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/util/olsc_File.h>

#define NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(rtype, methodInfo) \
template <typename EntryType> \
inline rtype CachedReaderWriter<EntryType>::methodInfo NN_NOEXCEPT

namespace nn { namespace olsc { namespace srv { namespace database {

// ---------------------------------------------------------------------------------
// Private
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, ReadImpl(int* outCount, EntryType out[], const char* path, int offset, int count))
{
    auto offsetByte = offset * sizeof(EntryType);
    auto countByte = count * sizeof(EntryType);

    size_t readSize;
    NN_RESULT_DO(util::ReadFile(&readSize, path, out, countByte, offsetByte));

    *outCount = static_cast<int>(readSize) / sizeof(EntryType);
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, WriteImpl(const char* path, const EntryType data[], int offset, int count))
{
    auto offsetByte = offset * sizeof(EntryType);
    auto countByte = count * sizeof(EntryType);

    auto modifyFile = [data, offsetByte, countByte](const char* path)
    {
        return util::ModifyFile(path, data, countByte, offsetByte);
    };

    NN_RESULT_TRY(modifyFile(path))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_DO(util::CreateFile(path, m_MaxEntryCountInFile * sizeof(EntryType)));
            NN_RESULT_DO(modifyFile(path));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, MoveToFront(const char* path, int toOffset, int fromOffset, int count))
{
    NN_SDK_ASSERT(toOffset < fromOffset);
    NN_SDK_ASSERT(fromOffset + count <= m_MaxEntryCountInFile);

    auto restMoveCount = count;
    auto readIndex = fromOffset;
    auto stride = fromOffset - toOffset;
    while (restMoveCount > 0)
    {
        auto readSize = std::min(restMoveCount, m_CacheBufferCount);
        NN_RESULT_DO(ReadToCache(path, readIndex, readSize));
        NN_RESULT_DO(WriteFromCache(path, readIndex - stride, m_CachedCount));

        restMoveCount -= m_CachedCount;
        readIndex += m_CachedCount;
    }

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, MoveToBack(const char* path, int toOffset, int fromOffset, int count))
{
    NN_SDK_ASSERT(toOffset > fromOffset);
    NN_SDK_ASSERT(toOffset + count <= m_MaxEntryCountInFile);

    auto restMoveCount = count;
    auto readBaseIndex = fromOffset + count;
    auto stride = toOffset - fromOffset;

    while (restMoveCount > 0)
    {
        auto readSize = std::min(restMoveCount, m_CacheBufferCount);
        NN_RESULT_DO(ReadToCache(path, readBaseIndex - readSize, readSize));
        NN_RESULT_DO(WriteFromCache(path, readBaseIndex - readSize + stride, m_CachedCount));

        restMoveCount -= m_CachedCount;
        readBaseIndex -= m_CachedCount;
    }

    NN_RESULT_SUCCESS;
}


// ---------------------------------------------------------------------------------
// Public
// ---------------------------------------------------------------------------------
NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(void, Clear())
{
    m_CachedCount = 0;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, WriteFromCache(const char* path, int offset, int count))
{
    NN_SDK_ASSERT(count <= m_CachedCount);

    std::lock_guard<decltype(m_CacheBufferLock)> lock(m_CacheBufferLock);

    NN_RESULT_DO(WriteImpl(path, m_CacheBuffer, offset, count));
    m_CachedOffset = offset;
    m_CachedCount = count;
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, ReadToCache(const char* path, int offset, int count))
{
    std::lock_guard<decltype(m_CacheBufferLock)> lock(m_CacheBufferLock);

    NN_RESULT_DO(ReadImpl(&m_CachedCount, m_CacheBuffer, path, offset, count));
    m_CachedOffset = offset;

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, Write(const char* path, const EntryType data[], int offset, int count))
{
    std::lock_guard<decltype(m_CacheBufferLock)> lock(m_CacheBufferLock);

    NN_RESULT_DO(WriteImpl(path, data, offset, count));

    auto copyBeginOffset = std::max(m_CachedOffset, offset);
    auto copyEndOffset = std::min(m_CachedOffset + m_CachedCount, offset + count);

    if (copyBeginOffset < copyEndOffset)
    {
        auto offsetInData = copyBeginOffset - offset;
        auto offsetInCache = copyBeginOffset - m_CachedOffset;

        auto copyCount = copyEndOffset - copyBeginOffset;
        std::memcpy(&m_CacheBuffer[offsetInCache], &data[offsetInData], sizeof(EntryType) * copyCount);
    }

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, Read(int* outCount, EntryType out[], const char* path, int offset, int count))
{
    std::lock_guard<decltype(m_CacheBufferLock)> lock(m_CacheBufferLock);

    int readOffset = offset;
    int restReadCount = std::min(count, m_MaxEntryCountInFile - readOffset);
    int readCount = 0;

    while (restReadCount)
    {
        if (!(readOffset >= m_CachedOffset && readOffset < m_CachedOffset + m_CachedCount))
        {
            // TODO:
            // readOffset + m_CacheBufferCount がファイルサイズを超える場合に
            // 超えた分だけ readOffset を前にずらすことでキャッシュに載る量を増やすことができる。
            NN_RESULT_DO(ReadImpl(&m_CachedCount, m_CacheBuffer, path, readOffset, m_CacheBufferCount));
            m_CachedOffset = readOffset;
        }

        auto offsetInCache = readOffset - m_CachedOffset;
        auto copyCount = std::min(restReadCount, m_CachedCount - offsetInCache);

        std::memcpy(&out[readCount], &m_CacheBuffer[offsetInCache], sizeof(EntryType) * copyCount);

        readCount += copyCount;
        readOffset += copyCount;
        restReadCount -= copyCount;
    }
    *outCount = readCount;
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD(Result, Move(const char* path, int toOffset, int fromOffset, int count))
{
    std::lock_guard<decltype(m_CacheBufferLock)> lock(m_CacheBufferLock);

    if (toOffset < fromOffset)
    {
        NN_RESULT_DO(MoveToFront(path, toOffset, fromOffset, count));
    }
    else if (toOffset > fromOffset)
    {
        NN_RESULT_DO(MoveToBack(path, toOffset, fromOffset, count));
    }

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::olsc::srv::database

#undef NN_OLSC_DEFINE_CACHED_READER_WRITER_METHOD
