﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  転送タスクに関連する型の定義です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_ApplicationId.h>
#include <nn/account/account_Types.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace olsc {

//------------------------------------------------------------------------------------

typedef Bit64   TransferTaskId;   //!<    転送タスクの識別子です。

/**
*   @brief  転送タスクの種類を表す列挙体です。
*/
enum class TransferTaskKind : Bit8 {
    Upload,         //!<    アップロードタスクを表す。
    Download,       //!<    ダウンロードタスクを表す。
};

/**
*   @brief  転送タスクの状態を表す列挙体です。
*/
enum class TransferTaskStatus : Bit8 {
    Runnable,       //!<    実行可能状態であることを表す。
    Running,        //!<    実行状態であることを表す。
    Error,          //!<    復旧不可能なエラー状態であることを表す。
    Completed,      //!<    タスクが正常終了していることを表す。
};

/**
*   @brief  転送タスクのランクを表す列挙体です。
*/
enum class TransferTaskRank : Bit8 {
    Basic       = 127,          //!<    標準的なランク
    Postponed   = 191,          //!<    後回しで実行されるランク。このランクの転送タスクは実行間隔にインターバルがあります。
};

/**
*   @brief  転送タスクの情報を表す構造体です。
*/
struct TransferTaskInfo {
    TransferTaskId          id;                 //!<    タスク ID。
    TransferTaskKind        type;               //!<    タスク種別。
    TransferTaskStatus      status;             //!<    タスクの状態。
    TransferTaskRank        rank;               //!<    タスクの階級。
    time::PosixTime         createdTime;        //!<    タスク作成時刻。
    account::Uid            uid;                //!<    対象のユーザ。
    ApplicationId           applicationId;      //!<    対象のアプリケーション。
};

/**
*   @brief  転送タスクの進捗を表す構造体です。
*/
struct TransferTaskProgress {
    size_t              totalSize;          //!<    総サイズ
    size_t              completedSize;      //!<    転送完了サイズ
    size_t              throughput;         //!<    スループット
};

/**
*   @brief  転送タスクのリトライ不可能なエラーを表す構造体です。
*/
struct TransferTaskErrorInfo {
    account::Uid        uid;                //!<    対象のユーザ。
    ApplicationId       applicationId;      //!<    対象のアプリケーション。
    TransferTaskKind    kind;               //!<    タスクの種別。
    bool                isRetryable;        //!<    true ならリトライ可能なタスク。
    time::PosixTime     registeredTime;     //!<    エラー発生時刻。
    Result              lastResult;         //!<    エラー発生時の Result
};

}} // namespace nn::olsc
