﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief      オンラインストレージの転送タスクリストに対する操作を行うインターフェースです。
 */

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/olsc/olsc_Stopper.h>
#include <nn/olsc/olsc_SystemEventHolder.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace olsc {
    namespace srv {
        class ITransferTaskListController;
        class IStopperObject;
    }

    class TransferTaskListController
    {
        NN_DISALLOW_COPY(TransferTaskListController);

    public:
        TransferTaskListController() NN_NOEXCEPT;
        ~TransferTaskListController() NN_NOEXCEPT;

        TransferTaskListController& operator=(TransferTaskListController&& rhs) NN_NOEXCEPT;
        TransferTaskListController(TransferTaskListController&& rhs) NN_NOEXCEPT;

        /**
        *       @brief      転送タスクの数を取得します。
        *
        *       @return     処理結果。
        *
        *       @details    転送タスクリストに登録されている転送タスクの数を取得します。
        *                   この関数を呼んだ後に List を呼び出すようなケースでは
        *                   バックグラウンドでの転送タスク追加処理によって、整合性が崩れる可能性があります。
        *
        *                   DaemonController::StopAutonomyTaskExecution() を使用することで、転送タスクの追加処理を止めることができます。
        */
        int GetTransferTaskCount() NN_NOEXCEPT;

        /**
        *       @brief      転送タスク情報を取得します。
        *
        *       @param[out] out         取得した転送タスク情報の格納先。
        *
        *       @return     転送タスクが存在する場合は true。
        *
        *       @pre
        *           -   out != nullptr
        *
        *       @details    id の一致する転送タスク情報を取得します。
        *
        */
        bool GetTransferTask(TransferTaskInfo* out, TransferTaskId id) NN_NOEXCEPT;

        /**
        *       @brief      転送タスク情報の一覧を取得します。
        *
        *       @param[out] outValue    取得した転送タスクの格納先。
        *       @param[in]  count       取得する最大数。
        *       @param[in]  offset      取得を開始する位置。
        *
        *       @return     取得した数。
        *
        *       @pre
        *           -   count >= 0
        *           -   outValue のサイズ >= count
        *           -   offset >= 0
        *
        *       @details    転送タスクリストに登録されている転送タスク情報の一覧を取得します。
        *                   この関数を複数回呼び出すようなケースでは
        *                   バックグラウンドでの転送タスク追加処理によって、整合性が崩れる可能性があります。
        *
        *                   DaemonController::StopAutonomyTaskExecution() 使用することで、転送タスクの追加処理を止めることができます。
        *
        */
        int ListTransferTask(TransferTaskInfo outValue[], int count, int offset) NN_NOEXCEPT;

        /**
        *   @brief      転送タスクを削除します。
        *
        *   @param[in]  id    対象の転送タスクID。
        *
        *   @return     処理結果を返します。
        *
        *   @details    id の一致する転送タスクを転送タスクリストから削除します。
        *               タスクが実行中の場合はそのタスク実行はキャンセルされます。
        *
        */
        Result DeleteTransferTask(TransferTaskId id) NN_NOEXCEPT;


        /**
        *   @brief      転送タスクの優先度を上げます。
        *
        *   @param[in]  id    対象の転送タスクID。
        *
        *   @return     処理結果を返します。
        *
        *   @details    id の一致する転送タスクの優先度を引き上げ直ちに実行します。
        *               あらかじめ実行されていたタスクは中断されます。
        *
        */
        Result RaiseTransferTaskPriority(TransferTaskId id) NN_NOEXCEPT;

        /**
        *   @brief      転送タスクの停止時にシグナルされる手動クリアイベントを取得します。
        *               この API は将来的に削除されます。代わりに GetTransferTaskFinishedEvent() を使用してください。
        *
        *   @param[out] out         手動クリアイベント。
        *
        *   @return     処理結果を返します。
        *
        *   @details    転送タスクが停止した際にシグナルされるイベントを取得します。
        *               停止の要因は問いません。（正常終了・リトライ不可エラー・リトライ可能エラー等全てを含みます。）
        *
        */
        Result GetTransferTaskEvent(SystemEventHolder* out) NN_NOEXCEPT;

        /**
        *   @brief      転送タスクの完了時にシグナルされる手動クリアイベントを取得します。
        *
        *   @return     イベント格納オブジェクトが返ります。
        *
        *   @details    転送タスクが完了した時にシグナルされるイベントを取得します。
        *               完了の要因は問わずにシグナルされます。（正常終了・リトライ不可エラー・リトライ可能エラー等）
        *
        */
        SystemEventHolder GetTransferTaskEndEvent() NN_NOEXCEPT;


        /**
        *   @brief      転送タスクの開始時にシグナルされる手動クリアイベントを取得します。
        *
        *   @return     イベント格納オブジェクトが返ります。
        *
        *   @details    転送タスクが開始された時にシグナルされるイベントを取得します。
        *
        */
        SystemEventHolder GetTransferTaskStartEvent() NN_NOEXCEPT;

        /**
        *   @brief      転送タスクの進捗を取得します。
        *
        *   @param[out] out         進捗。
        *   @param[in]  id          対象の転送タスクID。
        *
        *   @return     処理結果を返します。
        *
        *   @details    id の一致する転送タスクの進捗を取得します。
        *
        */
        Result GetTransferTaskProgress(TransferTaskProgress* out, TransferTaskId id) NN_NOEXCEPT;


        /**
        *   @brief      転送タスクの処理結果を取得します。
        *
        *   @param[in]  id          対象の転送タスクID。
        *
        *   @return     処理結果を返します。
        *
        *   @pre
        *       id で指定した転送タスクが存在し、状態が完了やエラーである
        *
        *   @details    id の一致する転送タスクの status が Error の場合に、この関数を呼ぶことでエラーの原因となった Result 値を取得することができます。
        *               id で指定したタスクが存在しないなど、事前条件を満たさない場合はこの関数は意味のある値を返さないことに注意してください。
        *
        */
        Result GetTransferTaskLastResult(TransferTaskId id) NN_NOEXCEPT;

        /**
        *   @brief      指定したタスクが実行中の場合にタスクを停止します。
        *
        *   @return     なし。
        *
        *   @details    指定した転送タスクが実行されていない場合は何もしません。
        *               タスクを停止しても StopNextTransferTaskExecution() で次のタスクの実行を禁止していない場合
        *               すぐにタスクが再開することに注意してください。
        *
        *               この関数は内部的にはタスクの停止を要求するだけで、タスクの停止を待たずに制御を戻します。
        *               タスクの停止を待つには、GetTransferTaskEndEvent で取得できるイベントのシグナルを待つようにしてください。
        *
        */
        void SuspendTransferTask(TransferTaskId tid) NN_NOEXCEPT;

        /**
        *       @brief      転送タスクの実行を停止します。
        *
        *       @details    この関数の返り値の Stopper オブジェクトが存在している限り、
        *                   新しい転送タスクが実行されることはありません。
        *
        *                   この関数呼び出しによって現在実行中のタスクは停止されません。
        *                   実行中のタスクを停止する場合は併せて SuspendTransferTask() を呼び出すようにしてください。
        *
        */
        Stopper StopNextTransferTaskExecution() NN_NOEXCEPT;

        /**
        *   @brief  この API は将来的に削除されます。代わりに ForbidTransferTaskExecution() を使用するようにして下さい。
        */
        Stopper StopTransferTaskExecution() NN_NOEXCEPT;
    private:
        friend void OpenTransferTaskListController(TransferTaskListController* out) NN_NOEXCEPT;
        friend TransferTaskListController OpenTransferTaskListController() NN_NOEXCEPT;
        friend void swap(TransferTaskListController& lhs, TransferTaskListController& rhs) NN_NOEXCEPT;
        NN_IMPLICIT TransferTaskListController(srv::ITransferTaskListController* interface) NN_NOEXCEPT;

        srv::ITransferTaskListController* m_Interface;
    };

    void swap(TransferTaskListController& lhs, TransferTaskListController& rhs) NN_NOEXCEPT;

}} // namespace nn::olsc
