﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/util/util_Optional.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/kvdb/kvdb_BoundedString.h>

namespace nn { namespace ns { namespace srv {

    struct VersionListDatabaseConfig
    {
        static const int    MaxEntryCount       = 16 * 1024;
        static const char*  DatabaseFileName;
        static const char*  HeaderFileName;
    };

    // -------- 必須バージョン記録用 --------
    struct RequiredVersionDatabaseConfig
    {
        static const int    MaxEntryCount       = 16 * 1024;
        static const char*  DatabaseFileName;
        static const char*  HeaderFileName;
    };

    // -------- 共通 --------
    struct VersionManagementDatabaseHeader
    {
        int32_t     formatVersion;
        char        signature[4];
        int64_t     lastModified;
        int32_t     entryCount;
        int32_t     maxEntryCount;

        static VersionManagementDatabaseHeader Make(int maxEntryCount) NN_NOEXCEPT;
        bool IsValid() NN_NOEXCEPT;
    };

    class VersionManagementDatabaseCommon
    {
    public :
        typedef kvdb::BoundedString<32> Path;
        static Path MakeFilePath(const char* mountName, const char* fileName) NN_NOEXCEPT;
        static Result RewriteFile(const char* fileName, const void* data, size_t dataSize) NN_NOEXCEPT;
        static Result ReadFile(const char* fileName, void* out, size_t readSize) NN_NOEXCEPT;
    };


    /*
        EntryType ... DB に格納するエントリの型
        必須メンバ
            - Bit64  id

        Config    ... DB の設定
        必須メンバ
            - static const int MaxEntryCount
            - static const char* DatabaseFileName
            - static const char* HeaderFileName
    */
    template<typename EntryType, typename Config>
    class VersionManagementDatabase
    {
    public:
        typedef EntryType SelfEntryType;
        VersionManagementDatabase(const char* mountName, void* buffer, size_t bufferSize) NN_NOEXCEPT;
        Result Load() NN_NOEXCEPT;
        Result Add(const EntryType& entry) NN_NOEXCEPT;
        EntryType* Find(Bit64 id) NN_NOEXCEPT;
        EntryType* FindOrDefault(Bit64 id, EntryType* defaultEntry) NN_NOEXCEPT;
        EntryType* Get(int index) NN_NOEXCEPT;
        Result Update(Bit64 id, const EntryType& entryType) NN_NOEXCEPT;
        void SetLastModified(int64_t lastModified) NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT;
        int32_t GetEntryCount() const NN_NOEXCEPT;

    private:
        VersionManagementDatabaseHeader m_Header;
        void* m_pBuffer;
        size_t m_BufferSize;

        typedef kvdb::BoundedString<32> Path;
        util::optional<Path> m_MountName;
    };

    typedef VersionManagementDatabase<VersionListEntry, VersionListDatabaseConfig> VersionListDatabase;
    typedef VersionManagementDatabase<RequiredVersionEntry, RequiredVersionDatabaseConfig> RequiredVersionDatabase;


    // --------

    class VersionListImporter
    {
    public:
        explicit VersionListImporter(VersionListDatabase* db) : m_pDb(db)
        {}

        template<typename InputStreamType>
        Result Import(InputStreamType& inputStream) NN_NOEXCEPT;
    private:
        VersionListDatabase* m_pDb;
    };

}}}


// ------ 実装

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ns/ns_Result.h>

namespace nn { namespace ns { namespace srv {


    template<typename EntryType, typename Config>
    VersionManagementDatabase<EntryType, Config>::VersionManagementDatabase(const char* mountName, void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_pBuffer(buffer), m_BufferSize(bufferSize)
    {
        NN_ABORT_UNLESS(m_BufferSize / sizeof(EntryType) >= Config::MaxEntryCount);
        m_MountName.emplace();
        m_MountName->Assign(mountName);

        m_Header = VersionManagementDatabaseHeader::Make(Config::MaxEntryCount);
        std::memset(m_pBuffer, 0, m_BufferSize);
    }

    template<typename EntryType, typename Config>
    Result VersionManagementDatabase<EntryType, Config>::Load() NN_NOEXCEPT
    {
        NN_RESULT_TRY(VersionManagementDatabaseCommon::ReadFile(
                        VersionManagementDatabaseCommon::MakeFilePath(
                            *m_MountName, Config::HeaderFileName), &m_Header, sizeof(m_Header)))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        NN_SDK_ASSERT(m_Header.IsValid()); // TODO: Result

        NN_ABORT_UNLESS(m_Header.entryCount <= Config::MaxEntryCount);
        NN_ABORT_UNLESS(m_Header.entryCount * sizeof(EntryType) <= m_BufferSize);

        NN_RESULT_DO(VersionManagementDatabaseCommon::ReadFile(
                        VersionManagementDatabaseCommon::MakeFilePath(
                            *m_MountName, Config::DatabaseFileName), m_pBuffer, m_Header.entryCount * sizeof(EntryType)));

        NN_RESULT_SUCCESS;
    }

    template<typename EntryType, typename Config>
    Result VersionManagementDatabase<EntryType, Config>::Commit() NN_NOEXCEPT
    {
        NN_RESULT_DO(
            VersionManagementDatabaseCommon::RewriteFile(
                VersionManagementDatabaseCommon::MakeFilePath(
                    *m_MountName, Config::HeaderFileName).Get(), &m_Header, sizeof(m_Header)));

        NN_RESULT_DO(
            VersionManagementDatabaseCommon::RewriteFile(
                VersionManagementDatabaseCommon::MakeFilePath(
                    *m_MountName, Config::DatabaseFileName).Get(), m_pBuffer, m_BufferSize));


        NN_RESULT_DO(nn::fs::CommitSaveData(*m_MountName));
        NN_RESULT_SUCCESS;
    }


    template<typename EntryType, typename Config>
    Result VersionManagementDatabase<EntryType, Config>::Add(const EntryType& entry) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            m_Header.entryCount < m_Header.maxEntryCount, ResultOutOfMaxVersionListCount());

        static_cast<EntryType*>(m_pBuffer)[m_Header.entryCount++] = entry;

        NN_RESULT_SUCCESS;
    }

    template<typename EntryType, typename Config>
    EntryType* VersionManagementDatabase<EntryType, Config>::FindOrDefault(Bit64 id, EntryType* defaultEntry) NN_NOEXCEPT
    {
        auto pEntry = static_cast<EntryType*>(m_pBuffer);
        for (int i = 0; i < m_Header.entryCount; ++i)
        {
            if (pEntry[i].id == id)
            {
                return &pEntry[i];
            }
        }
        return defaultEntry;
    }

    template<typename EntryType, typename Config>
    EntryType* VersionManagementDatabase<EntryType, Config>::Find(Bit64 id) NN_NOEXCEPT
    {
        return FindOrDefault(id, nullptr);
    }

    template<typename EntryType, typename Config>
    EntryType* VersionManagementDatabase<EntryType, Config>::Get(int index) NN_NOEXCEPT
    {
        if (index >= m_Header.entryCount) {
            return nullptr;
        }
        auto pEntry = static_cast<EntryType*>(m_pBuffer);
        return &pEntry[index];
    }

    template<typename EntryType, typename Config>
    Result VersionManagementDatabase<EntryType, Config>::Update(Bit64 id, const EntryType& entry) NN_NOEXCEPT
    {
        auto pEntry = Find(id);
        if (!pEntry)
        {
            return Add(entry);
        }

        *pEntry = entry;
        NN_RESULT_SUCCESS;
    }

    template<typename EntryType, typename Config>
    void VersionManagementDatabase<EntryType, Config>::SetLastModified(int64_t lastModified) NN_NOEXCEPT
    {
        m_Header.lastModified = lastModified;
    }

    template<typename EntryType, typename Config>
    int32_t VersionManagementDatabase<EntryType, Config>::GetEntryCount() const NN_NOEXCEPT
    {
        return m_Header.entryCount;
    }

}}}
