﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ns/ns_ApplicationDeliveryInfo.h>
#include <nn/ns/ns_ContentDeliveryProgress.h>
#include <nn/ns/ns_SystemDeliveryInfo.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns { namespace srv {

    class ApplicationRecordDatabase;
    class ApplicationVersionManager;
    class BlackListManager;
    class IntegratedContentManager;
    class CommitManager;
    class GameCardManager;
    class RequestServer;

    class ContentDeliveryManager
    {
    public:
        Result Initialize(ApplicationRecordDatabase* pRecordDb, ApplicationVersionManager* pVersionManager, BlackListManager* pBlackListManager, CommitManager* pCommitManager, GameCardManager* pGameCardManager, IntegratedContentManager* pIntegrated, RequestServer* pRequestServer) NN_NOEXCEPT;

        Result GetSystemDeliveryInfo(SystemDeliveryInfo* outValue) NN_NOEXCEPT;
        Result CompareSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT;
        Result SelectLatestSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo& receiverSystemInfo, const ApplicationDeliveryInfo receiverAppInfo[], int numAppList) NN_NOEXCEPT;
        Result VerifyDeliveryProtocolVersion(const SystemDeliveryInfo& info) NN_NOEXCEPT;

        Result GetApplicationDeliveryInfo(int* outCount, ApplicationDeliveryInfo* outList, int numList, ncm::ApplicationId id, ApplicationDeliveryAttribute attributes) NN_NOEXCEPT;
        Result HasAllContentsToDeliver(bool* outValue, const ApplicationDeliveryInfo appList[], int numList) NN_NOEXCEPT;
        Result CompareApplicationDeliveryInfo(int* outValue, const ApplicationDeliveryInfo lhsList[], int numLhsList, const ApplicationDeliveryInfo rhsList[], int numRhsList) NN_NOEXCEPT;
        Result CanDeliverApplication(bool* outValue, const ApplicationDeliveryInfo receiverList[], int numReceiverList, const ApplicationDeliveryInfo senderList[], int numSenderList) NN_NOEXCEPT;
        Result ListContentMetaKeyToDeliverApplication(int* outCount, ncm::ContentMetaKey outList[], int numList, int offset, const ApplicationDeliveryInfo infoList[], int numInfoList) NN_NOEXCEPT;
        Result NeedsSystemUpdateToDeliverApplication(bool* outValue, const ApplicationDeliveryInfo applicationList[], int numApplicationList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT;
        Result EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey keyList[], int numKey) NN_NOEXCEPT;

        Result RequestReceiveApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id, ncm::StorageId storageId) NN_NOEXCEPT;
        Result CommitReceiveApplication(ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetReceiveApplicationProgress(ReceiveApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result RequestSendApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetSendApplicationProgress(SendApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result ListNotCommittedContentMeta(int* outCount, ncm::ContentMetaKey outList[], int listCount, ncm::ApplicationId id, int offset) NN_NOEXCEPT;
        Result CreateDownloadTask(const ncm::ContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT;

        Result GetApplicationDeliveryInfoHash(ApplicationDeliveryInfoHash* outValue, const ApplicationDeliveryInfo infoList[], int numList) NN_NOEXCEPT;

        // SystemUpdateInterface から参照できるようにする
        static Result GetSystemUpdateMeta(ncm::ContentMetaKey* outValue, const SystemDeliveryInfo& info) NN_NOEXCEPT;

    private:
        static uint32_t GetSystemDeliveryProtocolVersion() NN_NOEXCEPT;
        static uint32_t GetApplicationDeliveryProtocolVersion() NN_NOEXCEPT;

        Result VerifySystemDeliveryInfo(const SystemDeliveryInfo& info, bool skipApplicationCompatibilityCheck = false) NN_NOEXCEPT;
        Result VerifyApplicationDeliveryInfo(const ApplicationDeliveryInfo& info) NN_NOEXCEPT;

        static Result VerifySystemDeliveryProtocolCompatibility(const SystemDeliveryInfo& info) NN_NOEXCEPT;
        static Result VerifyApplicationDeliveryProtocolCompatibility(const SystemDeliveryInfo& info) NN_NOEXCEPT;
        static Result VerifyApplicationDeliveryProtocolCompatibility(const ApplicationDeliveryInfo& info) NN_NOEXCEPT;
        static Result VerifyApplicationDeliveryProtocolCompatibilityImpl(uint32_t protocolVersion) NN_NOEXCEPT;

        static Result VerifyMacOfSystemDeliveryInfo(const SystemDeliveryInfo& info) NN_NOEXCEPT;
        static Result VerifyMacOfApplicationDeliveryInfo(const ApplicationDeliveryInfo& info) NN_NOEXCEPT;

        ApplicationRecordDatabase* m_pRecordDb;
        ApplicationVersionManager* m_pVersionManager;
        BlackListManager* m_pBlackListManager;
        CommitManager* m_pCommitManager;
        GameCardManager* m_pGameCardManager;
        IntegratedContentManager* m_pIntegrated;
        RequestServer* m_pRequestServer;
        uint32_t m_RequiredSystemVersion;
    };

}}}
