﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/kvdb/kvdb_FileKeyValueStore.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Span.h>
#include "ns_OsUtil.h"

namespace nn { namespace ns { namespace srv {

    namespace detail {
        struct ApplicationRecordIndex
        {
            int64_t eventCount;
            int count;
            Bit8 reserved[4];
            ApplicationRecord list[8192];

            int64_t GenerateEventCount()
            {
                eventCount++;
                return eventCount;
            }

            void Clear() NN_NOEXCEPT
            {
                eventCount = 0;
                count = 0;
                std::memset(reserved, 0, sizeof(reserved));
                std::memset(list, 0, sizeof(list));
            }

            Result Push(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT;
            Result Update(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT;
            Result UpdateWithoutReorder(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT;
            Result Delete(ncm::ApplicationId id) NN_NOEXCEPT;
            Result Get(ApplicationRecord* outValue, ncm::ApplicationId id) const NN_NOEXCEPT;
            Result SetAttribute(ncm::ApplicationId id, ApplicationRecordAttribute attribute, bool isOn) NN_NOEXCEPT;
            Result GetAttribute(bool* outValue, ncm::ApplicationId id, ApplicationRecordAttribute attribute) const NN_NOEXCEPT;
            bool Has(ncm::ApplicationId id) const NN_NOEXCEPT;

        private:
            Result FindIndex(int* outValue, ncm::ApplicationId id) const NN_NOEXCEPT;
        };

        struct ApplicationRecordData
        {
            int entryCount;
            Bit8 reserved[4];
            ApplicationRecordProperty property;
            ncm::StorageContentMetaKey list[2048];

            void Clear() NN_NOEXCEPT
            {
                entryCount = 0;
                std::memset(reserved, 0, sizeof(reserved));
                std::memset(list, 0, sizeof(0));
                std::memset(&property, 0, sizeof(property));
            }

            Result Push(const ncm::StorageContentMetaKey& key, bool allowOverwrite) NN_NOEXCEPT;

            bool HasAll(const ncm::ContentMetaKey keyList[], int count) const NN_NOEXCEPT;
            bool Has(const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
            util::optional<ncm::StorageContentMetaKey> GetById(Bit64 id) const NN_NOEXCEPT;

            template <class T>
            void Delete(const T& key) NN_NOEXCEPT;

            size_t CalculateSerializeSize() const NN_NOEXCEPT
            {
                return sizeof(entryCount) + sizeof(reserved) + sizeof(property) + sizeof(ncm::StorageContentMetaKey) * entryCount;
            }
        };
    }

    class ApplicationRecordAccessor;
    class ApplicationRecordDatabase
    {
    public:
        ApplicationRecordDatabase() : m_IsDataOccupied(false), m_SystemEvent(os::EventClearMode_AutoClear, true) {}

        Result Initialize(fs::SystemSaveDataId systemSaveDataId, IntegratedContentManager* pIntegrated) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        Result Push(ncm::ApplicationId id, ApplicationEvent event, const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite = true, bool skipNotify = false) NN_NOEXCEPT;
        Result UpdateEvent(ncm::ApplicationId id, ApplicationEvent event, bool doReorder = true, bool skipNotify = false) NN_NOEXCEPT;
        Result UpdateKey(ncm::ApplicationId id, ApplicationEvent event, const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite = true) NN_NOEXCEPT;
        Result Delete(ncm::ApplicationId id, bool skipNotify = false) NN_NOEXCEPT;

        Result Get(ApplicationRecord* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        int List(ApplicationRecord outList[], int count, int offset) NN_NOEXCEPT;
        int ListSpecificAttribute(ApplicationRecord outList[], int count, ApplicationRecordAttribute attribute) NN_NOEXCEPT;
        bool Has(ncm::ApplicationId id) NN_NOEXCEPT;
        Result HasContentMetaKey(bool* outHasRecord, bool* outInstalled, ncm::ContentMetaKey key, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetProperty(ApplicationRecordProperty* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result CountContentMeta(int* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result ListContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT;
        Result ListInstalledContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT;

        Result Open(ApplicationRecordAccessor* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        void Close(ApplicationRecordAccessor* accessor) NN_NOEXCEPT;

        Result FindMain(util::optional<ncm::ContentMetaKey>* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result FindLaunchableMain(
            util::optional<ncm::ContentMetaKey>* outValue,
            ncm::ApplicationId id) NN_NOEXCEPT;
        Result FindControl(util::optional<ncm::ContentMetaKey>* outCache, util::optional<ncm::ContentMetaKey>* outStorage, ncm::ApplicationId id) NN_NOEXCEPT;
        Result FindLaunchableApplicationAndPatch(
            util::optional<ncm::ContentMetaKey>* outApplication,
            util::optional<ncm::ContentMetaKey>* outPatch,
            ncm::ApplicationId id,
            ncm::StorageId appStorageId = ncm::StorageId::Any,
            ncm::StorageId patchStorageId = ncm::StorageId::Any) NN_NOEXCEPT;

        int64_t GenerateCount() NN_NOEXCEPT;

        Result EnableAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT;
        Result DisableAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT;
        Result IsAutoDeleteDisabled(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result EnableAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT;
        Result EnableAutoUpdate(ncm::ApplicationId id, const AutoUpdateInfo& info) NN_NOEXCEPT;
        Result DisableAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT;
        Result IsAutoUpdateEnabled(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetAutoUpdateInfo(AutoUpdateInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result SetApplicationTerminateResult(ncm::ApplicationId id, Result result) NN_NOEXCEPT;
        Result GetApplicationTerminateResult(Result* outValue, ncm::ApplicationId id, bool needsRaw) NN_NOEXCEPT;
        Result ClearApplicationTerminateResult(ncm::ApplicationId) NN_NOEXCEPT;

        Result RequestApplicationUpdate(ncm::ApplicationId id, Result result) NN_NOEXCEPT;
        Result IsApplicationUpdateRequested(util::optional<Result>* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result WithdrawApplicationUpdateRequest(ncm::ApplicationId id) NN_NOEXCEPT;

        Result RecommendCleanupAddOnContentsWithNoRights(ncm::ApplicationId id) NN_NOEXCEPT;
        Result WithdrawCleanupAddOnContentsWithNoRightsRecommendation(ncm::ApplicationId id) NN_NOEXCEPT;

        Result SetPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT;
        Result ClearPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT;
        Result ClearAllPreInstalledApplicationFlag() NN_NOEXCEPT;
        Result IsPreInstalledApplication(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        void NotifyUpdate() NN_NOEXCEPT
        {
            m_SystemEvent.Signal();
        }

        os::SystemEvent& GetSystemEvent() NN_NOEXCEPT
        {
            return m_SystemEvent;
        }

        void RegisterGameCardDatabase(ncm::ContentMetaDatabase&& db) NN_NOEXCEPT;
        void UnregisterGameCardDatabase() NN_NOEXCEPT;

        Result CleanupRedundant() NN_NOEXCEPT;

        Result DeleteKey(ncm::ApplicationId id, const ncm::StorageContentMetaKey& key) NN_NOEXCEPT;
        Result DeleteKey(ncm::ApplicationId id, const ncm::ContentMetaKey& key) NN_NOEXCEPT;

        // 他のスレッドによって ApplicationRecordAccessor が確保されているかを調べる
        // 自スレッドであれば問題ない
        bool IsLocked() const NN_NOEXCEPT
        {
            auto canLock = m_Mutex.TryLock();
            if (canLock)
            {
                m_Mutex.Unlock();
            }
            return !canLock;
        }

    private:
        detail::ApplicationRecordIndex m_Index;
        Result CreateIndex() NN_NOEXCEPT;
        Result LoadIndex() NN_NOEXCEPT;
        Result FlushIndex() NN_NOEXCEPT;

        kvdb::FileKeyValueStore m_Kvs;
        static const int KvsCacheBufferSize = 64 * 1024; // 1 アプリの記録最大サイズが 48KB で、それが収まるサイズ
        char m_KvsCacheBuffer[KvsCacheBufferSize];
        Result LoadKvs() NN_NOEXCEPT;
        Result CreateKvs() NN_NOEXCEPT;

        detail::ApplicationRecordData m_DataView;
        detail::ApplicationRecordData m_Data;
        util::optional<ncm::ContentMetaDatabase> m_GameCardDb;

        bool m_IsDataOccupied;

        Result CreateAndOpen(ApplicationRecordAccessor* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result ReadData(detail::ApplicationRecordData* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result WriteData(const detail::ApplicationRecordData& data, ncm::ApplicationId id) NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT;

        os::SystemEvent m_SystemEvent;
        mutable RecursiveMutex m_Mutex;

        IntegratedContentManager *m_pIntegrated {};
    };

    class ApplicationRecordAccessor
    {
        friend class ApplicationRecordDatabase;
    public:
        ApplicationRecordAccessor() NN_NOEXCEPT: m_Data(), m_DataView(), m_GameCardDb(), m_Db() {}
        ~ApplicationRecordAccessor() NN_NOEXCEPT
        {
            if (m_Data)
            {
                m_Db->Close(this);
            }
        }

        int Count() const NN_NOEXCEPT
        {
            return m_DataView->entryCount;
        }

        int GetDataCount() const NN_NOEXCEPT
        {
            return m_Data->entryCount;
        }

        bool HasAll(const ncm::ContentMetaKey keyList[], int count) const NN_NOEXCEPT
        {
            return m_DataView->HasAll(keyList, count);
        }

        bool Has(const ncm::ContentMetaKey& key) const NN_NOEXCEPT
        {
            return m_DataView->Has(key);
        }

        bool HasInstalled(const ncm::ContentMetaKey& key) const NN_NOEXCEPT
        {
            return m_Data->Has(key);
        }

        util::optional<ncm::StorageContentMetaKey> GetByIdFromData(Bit64 id) const NN_NOEXCEPT
        {
            return m_Data->GetById(id);
        }

        util::optional<ncm::StorageContentMetaKey> GetById(Bit64 id) const NN_NOEXCEPT
        {
            return m_DataView->GetById(id);
        }

        const ncm::StorageContentMetaKey& operator[](size_t i) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT(i < static_cast<size_t>(m_DataView->entryCount));

            return m_DataView->list[i];
        }

        const ncm::StorageContentMetaKey& GetDataAt(int i) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT(i < m_Data->entryCount);

            return m_Data->list[i];
        }

        const ApplicationRecordProperty& GetProperty() const NN_NOEXCEPT
        {
            return m_DataView->property;
        }

        util::Span<const ncm::StorageContentMetaKey> GetSpan() const NN_NOEXCEPT
        {
            const ncm::StorageContentMetaKey* begin = &m_DataView->list[0];
            return util::MakeSpan(begin, Count());
        }

        util::Span<const ncm::StorageContentMetaKey> GetDataSpan() const NN_NOEXCEPT
        {
            const ncm::StorageContentMetaKey* begin = &m_Data->list[0];
            return util::MakeSpan(begin, GetDataCount());
        }

    private:
        void Initialize(
            ncm::ApplicationId id,
            detail::ApplicationRecordData* data,
            detail::ApplicationRecordData* dataView,
            ncm::ContentMetaDatabase* gameCardDb,
            ApplicationRecordDatabase* db) NN_NOEXCEPT;

        Result Push(const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite) NN_NOEXCEPT;
        void Delete(const ncm::StorageContentMetaKey& key) NN_NOEXCEPT;
        void Delete(const ncm::ContentMetaKey& key) NN_NOEXCEPT;

        void RefreshView() NN_NOEXCEPT;

        const detail::ApplicationRecordData& GetData() const NN_NOEXCEPT
        {
            return *m_Data;
        }

        ncm::ApplicationId m_Id;
        detail::ApplicationRecordData* m_Data;
        detail::ApplicationRecordData* m_DataView;
        ncm::ContentMetaDatabase* m_GameCardDb;
        ApplicationRecordDatabase* m_Db;
    };


    util::optional<ncm::ContentMetaKey> DecideMainContentMetaKey(
        util::optional<ncm::ContentMetaKey> appKey,
        util::optional<ncm::ContentMetaKey> patchKey) NN_NOEXCEPT;

}}}
