﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/arp/arp_Api.h>
#include <nn/lr/lr_Path.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nim/nim_BackgroundDownloadStressTaskInfo.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationDeliveryInfo.h>
#include <nn/ns/ns_ContentDeliveryProgress.h>
#include <nn/ns/ns_ApplicationVerificationTypes.h>
#include <nn/ns/ns_SystemDeliveryInfo.h>
#include <nn/ns/detail/ns_IApplicationManagerInterface.sfdl.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/srv/ns_AccountProxyManager.h>
#include <nn/ns/srv/ns_AppletLauncher.h>
#include <nn/ns/srv/ns_ApplicationContentMetaStatusUtil.h>
#include <nn/ns/srv/ns_ApplicationControlDataManager.h>
#include <nn/ns/srv/ns_ApplicationCopyIdentifierManager.h>
#include <nn/ns/srv/ns_ApplicationDownloadManager.h>
#include <nn/ns/srv/ns_ApplicationEntityManager.h>
#include <nn/ns/srv/ns_ApplicationInstallRequest.h>
#include <nn/ns/srv/ns_ApplicationLaunchManager.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationVerificationManager.h>
#include <nn/ns/srv/ns_ApplicationVersionManager.h>
#include <nn/ns/srv/ns_ApplicationViewManager.h>
#include <nn/ns/srv/ns_BlackListManager.h>
#include <nn/ns/srv/ns_CommitManager.h>
#include <nn/ns/srv/ns_DataStore.h>
#include <nn/ns/srv/ns_ContentDeliveryManager.h>
#include <nn/ns/srv/ns_DownloadTaskListManager.h>
#include <nn/ns/srv/ns_DynamicRightsManager.h>
#include <nn/ns/srv/ns_ECommerceManager.h>
#include <nn/ns/srv/ns_FactoryResetManager.h>
#include <nn/ns/srv/ns_GameCardManager.h>
#include <nn/ns/srv/ns_GameCardRegistrationManager.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_IProgramLaunchObserver.h>
#include <nn/ns/srv/ns_LockedBufferManager.h>
#include <nn/ns/srv/ns_PreInstallManager.h>
#include <nn/ns/srv/ns_PseudoDeviceIdManager.h>
#include <nn/ns/srv/ns_PushNotificationDisptcher.h>
#include <nn/ns/srv/ns_RequestServerManager.h>
#include <nn/ns/srv/ns_RightsEnvironment.h>
#include <nn/ns/srv/ns_SdCardManager.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/ns/srv/ns_ShutdownManager.h>
#include <nn/ns/srv/ns_SystemReportManager.h>
#include <nn/ns/srv/ns_TicketManager.h>
#include <nn/ns/srv/ns_UserResourceManagerImpl.h>
#include <nn/ns/srv/ns_VulnerabilityManager.h>
#include <nn/os/os_Mutex.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ns/detail/ns_Log.h>

namespace nn { namespace ns { namespace srv {

    class ApplicationManagerInterfaceServer
        : public IProgramLaunchObserver
    {
    public:
        ApplicationManagerInterfaceServer() NN_NOEXCEPT;
        ~ApplicationManagerInterfaceServer() NN_NOEXCEPT;

        Result Initialize(RequestServer* reqestServerForSystemUpdate, SystemReportManager* systemReportManager) NN_NOEXCEPT;

        Result CalculateApplicationOccupiedSize(sf::Out<ns::ApplicationOccupiedSize> outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetApplicationContentPath(sf::Out<ncm::Path> outValue, ncm::ApplicationId id, ncm::ContentType type) NN_NOEXCEPT;
        Result ResolveApplicationContentPath(ncm::ProgramId id, ncm::ContentType type) NN_NOEXCEPT;
        Result TerminateApplication(os::ProcessId pid) NN_NOEXCEPT;
        Result BeginInstallApplication(ncm::ApplicationId id, std::uint32_t version, ncm::StorageId storage) NN_NOEXCEPT;
        Result PushDownloadTaskList(const sf::InBuffer& buffer) NN_NOEXCEPT;
        Result GetApplicationLogoData(sf::Out<std::int64_t> outSize, const sf::OutBuffer& outBuffer, ncm::ApplicationId id, const ncm::Path& logoPath) NN_NOEXCEPT;
        Result StartApplyDeltaTask(ncm::ApplicationId id) NN_NOEXCEPT;

        Result CountApplicationContentMeta(sf::Out<std::int32_t> outValue, ncm::ApplicationId appId) NN_NOEXCEPT;
        Result GetTotalSpaceSize(sf::Out<std::int64_t> outSize, ncm::StorageId storageId) NN_NOEXCEPT;
        Result GetFreeSpaceSize(sf::Out<std::int64_t> outSize, ncm::StorageId storageId) NN_NOEXCEPT;
        Result GetApplicationDesiredLanguage(sf::Out<Bit8> outValue, Bit32 supportedLanguageFlag) NN_NOEXCEPT;
        Result SetApplicationTerminateResult(ncm::ApplicationId id, uint32_t result) NN_NOEXCEPT;
        Result ClearApplicationTerminateResult(ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetApplicationTerminateResult(sf::Out<uint32_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetRawApplicationTerminateResult(sf::Out<uint32_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result ConvertApplicationLanguageToLanguageCode(sf::Out<settings::LanguageCode> outValue, Bit8 language) NN_NOEXCEPT;
        Result ConvertLanguageCodeToApplicationLanguage(sf::Out<Bit8> outValue, settings::LanguageCode code) NN_NOEXCEPT;
        Result GetBackgroundDownloadStressTaskInfo(sf::Out<nim::BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT;
        Result GetBackgroundApplyDeltaStressTaskInfo(sf::Out<nim::BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT;
        Result IsSystemProgramInstalled(sf::Out<bool> outValue, ncm::SystemProgramId id) NN_NOEXCEPT;

        Result GetRequestServerStopper(sf::Out<sf::SharedPointer<ns::detail::IRequestServerStopper>> outValue) NN_NOEXCEPT;
        Result GetOwnedApplicationContentMetaStatus(sf::Out<ns::ApplicationContentMetaStatus> outValue, ncm::ApplicationId appId, nn::Bit64 contentMetaId) NN_NOEXCEPT;
        Result RegisterContentsExternalKey(ncm::ApplicationId appId, nn::Bit64 contentMetaId) NN_NOEXCEPT;

        Result GetStorageSize(sf::Out<std::int64_t> outTotalSize, sf::Out<std::int64_t> outFreeSpaceSize, ncm::StorageId strogeId) NN_NOEXCEPT;

        Result GetRunningApplicationProgramId(sf::Out<ncm::ProgramId> outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetMainApplicationProgramIndex(sf::Out<uint8_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetMainApplicationProgramIndexByApplicationLaunchInfo(sf::Out<uint8_t> outValue, const ApplicationLaunchInfo& info) NN_NOEXCEPT;

        // ns_ApplicationViewApi.h
        Result GetApplicationViewDeprecated(const sf::OutArray<ns::ApplicationViewDeprecated>& outList, const sf::InArray<ncm::ApplicationId>& idList) NN_NOEXCEPT
        {
            return m_ViewManager.GetApplicationViewDeprecated(outList, idList);
        }
        Result GetApplicationView(const sf::OutArray<ns::ApplicationView>& outList, const sf::InArray<ncm::ApplicationId>& idList) NN_NOEXCEPT
        {
            return m_ViewManager.GetApplicationView(outList, idList);
        }
        Result GetApplicationViewDownloadErrorContext(nn::sf::Out<nn::err::ErrorContext> outValue, nn::ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ViewManager.GetApplicationViewDownloadErrorContext(outValue, id);
        }
        Result ListApplicationDownloadingContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, ncm::ApplicationId id, std::int32_t offset) NN_NOEXCEPT
        {
            return m_ViewManager.ListApplicationDownloadingContentMeta(outCount, outList, id, offset);
        }

        // ns_ApplicationRecordApi.h
        Result ListApplicationRecord(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationRecord>& outList, int offset) NN_NOEXCEPT
        {
            *outCount = m_RecordDb.List(outList.GetData(), static_cast<int>(outList.GetLength()), offset);
            NN_RESULT_SUCCESS;
        }
        Result GenerateApplicationRecordCount(sf::Out<std::int64_t> outCount) NN_NOEXCEPT
        {
            *outCount = m_RecordDb.GenerateCount();
            NN_RESULT_SUCCESS;
        }
        Result GetApplicationRecordUpdateSystemEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
        {
            *outValue = sf::NativeHandle(m_RecordDb.GetSystemEvent().GetReadableHandle(), false);
            NN_RESULT_SUCCESS;
        }
        Result DisableApplicationAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.DisableAutoDelete(id);
        }
        Result EnableApplicationAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.EnableAutoDelete(id);
        }
        Result TouchApplication(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.UpdateEvent(id, ApplicationEvent::Touched);
        }
        Result RequestApplicationUpdate(ncm::ApplicationId id, Bit32 resultInnerValue) NN_NOEXCEPT
        {
            return m_RecordDb.RequestApplicationUpdate(id, result::detail::ConstructResult(resultInnerValue));
        }
        Result IsApplicationUpdateRequested(sf::Out<bool> outValue, sf::Out<Bit32> outInnerValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            util::optional<Result> result;
            NN_RESULT_DO(m_RecordDb.IsApplicationUpdateRequested(&result, id));
            *outValue = result;
            *outInnerValue = result ? result->GetInnerValueForDebug() : 0;

            NN_RESULT_SUCCESS;
        }
        Result WithdrawApplicationUpdateRequest(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.WithdrawApplicationUpdateRequest(id);
        }
        Result WithdrawCleanupAddOnContentsWithNoRightsRecommendation(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.WithdrawCleanupAddOnContentsWithNoRightsRecommendation(id);
        }
        Result HasApplicationRecord(sf::Out<bool> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            *outValue = m_RecordDb.Has(id);
            NN_RESULT_SUCCESS;
        }

        // ns_ApplicationRecordSystemApi.h
        Result GetApplicationRecord(sf::Out<ApplicationRecord> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.Get(outValue.GetPointer(), id);
        }
        Result PushApplicationRecord(ncm::ApplicationId id, ns::ApplicationEvent appEvent, const sf::InArray<ncm::StorageContentMetaKey>& keyList) NN_NOEXCEPT
        {
            return m_RecordDb.Push(id, appEvent, keyList.GetData(), static_cast<int>(keyList.GetLength()));
        }
        Result ListApplicationRecordContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, ncm::ApplicationId id, int offset) NN_NOEXCEPT
        {
            return m_RecordDb.ListContentMeta(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), id, offset);
        }
        Result ListApplicationRecordInstalledContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, ncm::ApplicationId id, int offset) NN_NOEXCEPT
        {
            return m_RecordDb.ListInstalledContentMeta(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), id, offset);
        }
        Result DeleteApplicationRecord(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.Delete(id);
        }
        Result GetApplicationRecordProperty(sf::Out<ns::ApplicationRecordProperty> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.GetProperty(outValue.GetPointer(), id);
        }
        Result EnableApplicationAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.EnableAutoUpdate(id);
        }
        Result DisableApplicationAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.DisableAutoUpdate(id);
        }
        Result SetPreInstalledApplication(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.SetPreInstalledApplicationFlag(id);
        }
        Result ClearPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_RecordDb.ClearPreInstalledApplicationFlag(id);
        }

        // ns_ApplicationControlDataApi.h
        Result GetApplicationControlData(sf::Out<std::uint32_t> outValue, const sf::OutBuffer& buffer, ns::ApplicationControlSource source, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ControlDataManager.GetApplicationControlData(outValue, buffer, source, id);
        }
        Result GetApplicationControlProperty(sf::Out<ApplicationControlProperty> outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
        {
            return m_ControlDataManager.GetApplicationControlProperty(outValue.GetPointer(), info, programIndex);
        }
        Result InvalidateAllApplicationControlCache() NN_NOEXCEPT
        {
            return m_ControlDataManager.InvalidateAllApplicationControlCache();
        }
        Result InvalidateApplicationControlCache(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ControlDataManager.InvalidateApplicationControlCache(id);
        }
        Result RequestDownloadApplicationControlData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ControlDataManager.RequestDownloadApplicationControlData(outHandle, outAsync, id);
        }
        Result GetMaxApplicationControlCacheCount(sf::Out<std::int32_t> outValue) NN_NOEXCEPT
        {
            return m_ControlDataManager.GetMaxApplicationControlCacheCount(outValue);
        }
        Result ListApplicationControlCacheEntryInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationControlCacheEntryInfo>& outList) NN_NOEXCEPT
        {
            return m_ControlDataManager.ListApplicationControlCacheEntryInfo(outCount, outList);
        }

        // ns_SdCardApi.h
        Result CleanupSdCard() NN_NOEXCEPT
        {
            return m_SdCardManager.Cleanup();
        }
        Result CheckSdCardMountStatus() NN_NOEXCEPT
        {
            return m_SdCardManager.CheckMountStatus();
        }
        Result GetSdCardMountStatusChangedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
        {
            return m_SdCardManager.GetMountStatusChangedEvent(outValue);
        }
        Result GetSdCardRemovedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
        {
            return m_SdCardManager.GetRemovedEvent(outValue);
        }
        Result GetLastSdCardMountUnexpectedResult() NN_NOEXCEPT
        {
            return m_SdCardManager.GetLastMountUnexpectedResult();
        }
        Result FormatSdCard() NN_NOEXCEPT
        {
            return m_SdCardManager.Format();
        }
        Result NeedsSystemUpdateToFormatSdCard(sf::Out<bool> outValue) NN_NOEXCEPT
        {
            return m_SdCardManager.NeedsSystemUpdateToFormat(outValue.GetPointer());
        }
        Result GetLastSdCardFormatUnexpectedResult() NN_NOEXCEPT
        {
            return m_SdCardManager.GetLastFormatUnexpectedResult();
        }
        Result InsertSdCard() NN_NOEXCEPT
        {
            m_SdCardManager.Insert();
            NN_RESULT_SUCCESS;
        }
        Result RemoveSdCard() NN_NOEXCEPT
        {
            m_SdCardManager.Remove();
            NN_RESULT_SUCCESS;
        }

        // ns_GameCardApi.h
        Result GetGameCardUpdateDetectionEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
        {
            auto handle = os::GetReadableHandleOfSystemEvent(m_GameCardManager.GetUpdateDetectionEvent().GetBase());
            *outValue = sf::NativeHandle(handle, false);
            NN_RESULT_SUCCESS;
        }
        Result GetGameCardAttachmentEvent(sf::Out<sf::NativeHandle> outEvent) NN_NOEXCEPT
        {
            auto handle = m_GameCardManager.GetAttachmentEvent();
            *outEvent = sf::NativeHandle(handle, false);
            NN_RESULT_SUCCESS;
        }
        Result GetGameCardAttachmentInfo(sf::Out<GameCardAttachmentInfo> outInfo) NN_NOEXCEPT
        {
            GameCardAttachmentInfo info;
            m_GameCardManager.GetAttachmentInfo(&info);
            *outInfo = info;
            NN_RESULT_SUCCESS;
        }
        Result GetGameCardStopper(sf::Out<sf::SharedPointer<ns::detail::IGameCardStopper>> outValue) NN_NOEXCEPT
        {
            *outValue = m_GameCardManager.GetStopper();
            NN_RESULT_SUCCESS;
        }
        Result GetGameCardMountFailureEvent(sf::Out<sf::NativeHandle> outEvent) NN_NOEXCEPT
        {
            auto handle = m_GameCardManager.GetMountFailureEvent();
            *outEvent = sf::NativeHandle(handle, false);
            NN_RESULT_SUCCESS;
        }
        Result IsGameCardInserted(sf::Out<bool> outValue) NN_NOEXCEPT
        {
            *outValue = m_GameCardManager.IsGameCardInserted();
            NN_RESULT_SUCCESS;
        }
        Result EnsureGameCardAccess() NN_NOEXCEPT
        {
            return m_GameCardManager.EnsureAccess();
        }
        Result GetLastGameCardMountFailureResult() NN_NOEXCEPT
        {
            return m_GameCardManager.GetLastMountFailureResult();
        }
        Result ListApplicationIdOnGameCard(sf::Out<int> outCount, const sf::OutArray<ncm::ApplicationId>& outList) NN_NOEXCEPT
        {
            return m_GameCardManager.ListApplicationId(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()));
        }

        // ns_GameCardRegistrationApi.h
        nn::Result RequestCheckGameCardRegistration(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ns::detail::IAsyncResult>> outAsync, nn::ncm::ApplicationId appId) NN_NOEXCEPT
        {
            ApplicationRecord record;
            NN_RESULT_DO(m_RecordDb.Get(&record, appId));
            NN_RESULT_DO(m_GameCardRegistrationManger.RequestCheckGameCardRegistration(outHandle, outAsync, appId));
            NN_RESULT_SUCCESS;
        }

        nn::Result RequestGameCardRegistrationGoldPoint(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ns::detail::IAsyncValue>> outAsync, const nn::account::Uid& uid, nn::ncm::ApplicationId appId) NN_NOEXCEPT
        {
            ApplicationRecord record;
            NN_RESULT_DO(m_RecordDb.Get(&record, appId));
            NN_RESULT_DO(m_GameCardRegistrationManger.RequestGameCardRegistrationGoldPoint(outHandle, outAsync, uid, appId));
            NN_RESULT_SUCCESS;
        }

        nn::Result RequestRegisterGameCard(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ns::detail::IAsyncResult>> outAsync, const nn::account::Uid& uid, nn::ncm::ApplicationId appId, std::int32_t goldPoint) NN_NOEXCEPT
        {
            ApplicationRecord record;
            NN_RESULT_DO(m_RecordDb.Get(&record, appId));
            NN_RESULT_DO(m_GameCardRegistrationManger.RequestRegisterGameCard(outHandle, outAsync, uid, appId, goldPoint));
            NN_RESULT_SUCCESS;
        }

        // ns_DownloadTaskApi.h
        Result RequestDownloadTaskList() NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.RequestDownloadTaskList(nim::ETag::MakeEmpty(), true);
        }
        Result RequestEnsureDownloadTask(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
        {
            return m_DownloadTaskListManager.RequestEnsureDownloadTask(outHandle, outAsync, m_RequestServer.Stop());
        }
        Result RequestDownloadTaskListData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
        {
            return m_DownloadTaskListManager.RequestDownloadTaskListData(outHandle, outAsync, m_RequestServer.Stop());
        }
        Result ClearTaskStatusList() NN_NOEXCEPT
        {
            return m_DownloadTaskListManager.Clear();
        }
        Result ListDownloadTaskStatus(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::DownloadTaskStatus>& outList) NN_NOEXCEPT
        {
            *outCount = m_DownloadTaskListManager.ListDownloadTaskStatus(outList.GetData(), static_cast<int>(outList.GetLength()));
            NN_RESULT_SUCCESS;
        }
        Result TryCommitCurrentApplicationDownloadTask() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());
            bool isCommittable;
            NN_RESULT_DO(m_CommitManager.IsCurrentApplicationDownloadTaskDynamicallyCommittable(&isCommittable));
            if (isCommittable)
            {
                auto stopper = m_RequestServer.Stop();
                NN_RESULT_DO(m_CommitManager.CommitCurrentApplicationDownloadTaskDynamically());
            }
            NN_RESULT_SUCCESS;
        }

        void EnableAutoCommit() NN_NOEXCEPT
        {
            m_CommitManager.EnableAutoCommit();
        }

        void DisableAutoCommit() NN_NOEXCEPT
        {
            m_CommitManager.DisableAutoCommit();
        }

        Result TriggerDynamicCommitEvent() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());
            NN_RESULT_DO(m_CommitManager.TriggerDynamicCommitEvent());

            NN_RESULT_SUCCESS;
        }

        // ns_ApplicationVersionSystemApi.h
        Result RequestVersionList() NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.RequestVersionList(nim::ETag::MakeEmpty());
        }
        Result UpdateVersionList(const sf::InBuffer& buffer) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());
            return m_ApplicationVersionManager.UpdateVersionList(buffer.GetPointerUnsafe(), buffer.GetSize());
        }
        Result PushLaunchVersion(ncm::ApplicationId id, std::uint32_t version) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.PushLaunchVersion(id, version);
        }
        Result ListRequiredVersion(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::RequiredVersionEntry>& requiredVersionEntries) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.ListRequiredVersion(outCount.GetPointer(), requiredVersionEntries.GetData(), static_cast<int>(requiredVersionEntries.GetLength()));
        }
        Result ListVersionList(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::VersionListEntry>& versionListEntries) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.ListVersionList(outCount.GetPointer(), versionListEntries.GetData(), static_cast<int>(versionListEntries.GetLength()));
        }
        Result RequestVersionListData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.RequestVersionListData(outHandle, outAsync, m_RequestServer.Stop());
        }
        Result PerformAutoUpdate() NN_NOEXCEPT
        {
            auto stopper = m_RequestServer.Stop();
            return m_ApplicationVersionManager.PerformAutoUpdate();
        }

        // ns_VerifyApplicationApi.h
        Result RequestVerifyApplicationDeprecated(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IProgressAsyncResult>> outAsync, ncm::ApplicationId appId, sf::NativeHandle transferHandle, uint64_t transferSize) NN_NOEXCEPT
        {
            return m_ApplicationVerificationManager.RequestVerifyApplication(outHandle, outAsync, appId, VerifyContentFlag_All._storage[0], std::move(transferHandle), transferSize);
        }
        Result RequestVerifyApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IProgressAsyncResult>> outAsync, ncm::ApplicationId appId, nn::Bit32 flags, sf::NativeHandle transferHandle, uint64_t transferSize) NN_NOEXCEPT
        {
            return m_ApplicationVerificationManager.RequestVerifyApplication(outHandle, outAsync, appId, flags, std::move(transferHandle), transferSize);
        }
        Result CorruptApplicationForDebug(ncm::ApplicationId id, Bit32 flags, ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());
            return m_ApplicationVerificationManager.CorruptApplicationForDebug(id, flags, storageId);
        }
        Result RequestVerifyAddOnContentsRights(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IProgressAsyncResult>> outAsync, ncm::ApplicationId appId) NN_NOEXCEPT
        {
            return m_ApplicationVerificationManager.RequestVerifyAddOnContentsRights(outHandle, outAsync, appId);
        }
        Result CorruptContentForDebug(const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());
            return m_ApplicationVerificationManager.CorruptContentForDebug(key, storageId);
        }

        // ns_VulnerabilityApi.h
        Result NeedsUpdateVulnerability(sf::Out<bool> outValue) NN_NOEXCEPT
        {
            outValue.Set(m_VulnerabilityManager.NeedsUpdateVulnerability());
            NN_RESULT_SUCCESS;
        }

        Result UpdateSafeSystemVersionForDebug(nn::ncm::SystemUpdateId systemUpdateId, std::uint32_t version) NN_NOEXCEPT
        {
            return m_VulnerabilityManager.UpdateSafeSystemVersionForDebug(systemUpdateId, version);
        }

        Result GetSafeSystemVersion(sf::Out<nn::ncm::ContentMetaKey> outValue) NN_NOEXCEPT
        {
            m_VulnerabilityManager.GetSafeSystemVersion(outValue.GetPointer());
            NN_RESULT_SUCCESS;
        }

        // ns_ApplicationEntityApi.h
        Result DeleteApplicationEntity(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteApplicationEntity(id);
        }
        Result DeleteApplicationCompletely(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteApplicationCompletely(id);
        }
        Result IsAnyApplicationEntityRedundant(sf::Out<bool> outValue) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.IsAnyApplicationEntityRedundant(outValue.GetPointer());
        }
        Result DeleteRedundantApplicationEntity() NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteRedundantApplicationEntity();
        }
        Result IsApplicationEntityMovable(sf::Out<bool> outValue, ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.IsApplicationEntityMovable(outValue.GetPointer(), id, storage);
        }
        Result MoveApplicationEntity(ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.MoveApplicationEntity(id, storage);
        }
        Result IsAnyApplicationEntityInstalled(sf::Out<bool> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.IsAnyApplicationEntityInstalled(outValue.GetPointer(), id);
        }
        Result DeleteApplicationContentEntities(ncm::ApplicationId id, uint32_t innerFlagsValue, ncm::StorageId storageId) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteApplicationContentEntities(id, innerFlagsValue, storageId);
        }
        Result CleanupUnrecordedApplicationEntity(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.CleanupUnrecordedApplicationEntity(id);
        }
        Result CleanupAddOnContentsWithNoRights(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.CleanupAddOnContentsWithNoRights(id);
        }
        Result DeleteApplicationCompletelyForDebug(ncm::ApplicationId id, uint32_t innerFlagsValue) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteApplicationCompletelyForDebug(id, innerFlagsValue);
        }

        // ns_ShutdownApi.h
        Result PrepareShutdown() NN_NOEXCEPT
        {
            return m_ShutdownManager.PrepareShutdown();
        }

        // ns_ApplicationLaunchManager.h
        Result LaunchApplication(sf::Out<os::ProcessId> outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.LaunchApplication(outValue.GetPointer(), info, programIndex);
        }
        Result DeprecatedLaunchApplication(sf::Out<os::ProcessId> outValue, ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT
        {
            ApplicationLaunchInfo info;
            NN_RESULT_DO(m_ApplicationLaunchManager.GetApplicationLaunchInfo(&info, id));
            NN_RESULT_DO(m_ApplicationLaunchManager.LaunchApplication(outValue.GetPointer(), info, programIndex));
            NN_RESULT_SUCCESS;
        }
        Result CheckApplicationLaunchVersion(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.CheckApplicationLaunchVersion(id);
        }
        Result CheckApplicationLaunchRights(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.CheckApplicationLaunchRights(id);
        }
        Result CheckApplicationResumeRights(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.CheckApplicationResumeRights(id);
        }
        Result GetDynamicCommitEvent(sf::Out<sf::NativeHandle> outHandle) NN_NOEXCEPT
        {
            *outHandle = sf::NativeHandle(m_CommitManager.GetDynamicCommitEvent().GetReadableHandle(), false);
            NN_RESULT_SUCCESS;
        }
        Result BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.BoostSystemMemoryResourceLimit(boostSize);
        }
        Result GetApplicationLaunchInfo(sf::Out<ApplicationLaunchInfo> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.GetApplicationLaunchInfo(outValue.GetPointer(), id);
        }
        Result AcquireApplicationLaunchInfo(sf::Out<ApplicationLaunchInfo> outValue, os::ProcessId processId) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.AcquireApplicationLaunchInfo(outValue.GetPointer(), processId);
        }

        void EnableApplicationCrashReport(bool isEnabled) NN_NOEXCEPT
        {
            nn::ns::srv::EnableApplicationCrashReport(isEnabled);
        }
        bool IsApplicationCrashReportEnabled() NN_NOEXCEPT
        {
            return nn::ns::srv::IsApplicationCrashReportEnabled();
        }

        void EnableApplicationAllThreadDumpOnCrash(bool isEnabled) NN_NOEXCEPT
        {
            nn::ns::srv::EnableApplicationAllThreadDumpOnCrash(isEnabled);
        }

        Result CalculateUserSaveDataStatistics(sf::Out<ns::UserSaveDataStatistics> pOut, const account::Uid& uid) NN_NOEXCEPT
        {
            return m_UserResourceManager.CalculateUserSaveDataStatistics(pOut, uid);
        }
        Result DeleteUserSaveDataAll(sf::Out<sf::SharedPointer<ns::detail::IProgressMonitorForDeleteUserSaveDataAll>> pOut, const account::Uid& uid) NN_NOEXCEPT
        {
            return m_UserResourceManager.DeleteUserSaveDataAll(pOut, uid);
        }
        Result DeleteUserSystemSaveData(const account::Uid& uid, fs::SystemSaveDataId sysSaveId) NN_NOEXCEPT
        {
            return m_UserResourceManager.DeleteUserSystemSaveData(uid, sysSaveId);
        }
        Result UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT
        {
            auto stopper = m_RequestServer.Stop();
            return m_UserResourceManager.UnregisterNetworkServiceAccount(uid);
        }
        Result UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT
        {
            auto stopper = m_RequestServer.Stop();
            return m_UserResourceManager.UnregisterNetworkServiceAccountWithUserSaveDataDeletion(uid);
        }

        // ns_FactoryResetApi.h
        Result ResetToFactorySettings() NN_NOEXCEPT
        {
            return m_FactoryResetManager.ResetToFactorySettings();
        }
        Result ResetToFactorySettingsWithoutUserSaveData() NN_NOEXCEPT
        {
            return m_FactoryResetManager.ResetToFactorySettingsWithoutUserSaveData();
        }
        Result ResetToFactorySettingsForRefurbishment() NN_NOEXCEPT
        {
            return m_FactoryResetManager.ResetToFactorySettingsForRefurbishment();
        }

        // ns_PseudoDeviceIdManager.h
        Result GetSystemSeedForPseudoDeviceId(sf::Out<ns::SystemSeedForPseudoDeviceId> pOutSeed) NN_NOEXCEPT
        {
            return m_PseudoDeviceIdManager.GetSystemSeed( pOutSeed.GetPointer() );
        }
        Result ResetSystemSeedForPseudoDeviceId() NN_NOEXCEPT
        {
            return m_PseudoDeviceIdManager.ResetSystemSeed();
        }

        // ns_Notification[System]Api.h
        Result IsNotificationSetupCompleted(sf::Out<bool> outValue) NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.IsNotificationSetupCompleted(outValue);
        }
        Result GetLastNotificationInfoCount(sf::Out<int64_t> outValue) NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.GetLastNotificationInfoCount(outValue);
        }
        Result ListLastNotificationInfo(sf::Out<int> outCount, const sf::OutArray<ns::NotificationInfo>& outList) NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.ListLastNotificationInfo(outCount, outList);
        }
        Result ListNotificationTask(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ns::AsyncTask>& outList) NN_NOEXCEPT
        {
            return m_PushNotificationDispatcher.ListNotificationTask(outCount, outList);
        }

        // ns_ApplicationContentMeta[System]Api.h
        Result ListApplicationContentMetaStatus(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationContentMetaStatus>& outList, ncm::ApplicationId appId, std::int32_t offset) NN_NOEXCEPT
        {
            return ListApplicationContentMetaStatusImpl(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), appId, offset, m_RecordDb, m_Integrated, false);
        }
        Result ListApplicationContentMetaStatusWithRightsCheck(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationContentMetaStatus>& outList, ncm::ApplicationId appId, std::int32_t offset) NN_NOEXCEPT
        {
            return ListApplicationContentMetaStatusImpl(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), appId, offset, m_RecordDb, m_Integrated, true);
        }
        Result ListAvailableAddOnContent(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationContentMetaStatus>& outList, ncm::ApplicationId appId, std::int32_t offset) NN_NOEXCEPT
        {
            return ListAvailableAddOnContentImpl(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), appId, offset, m_RecordDb, m_Integrated, m_ApplicationLaunchManager);
        }
        Result GetContentMetaStorage(sf::Out<ncm::StorageId> outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
        {
            return m_Integrated.GetContentMetaStorage(outValue.GetPointer(), key);
        }

        // ns_DeviceLinkApi.h
        bool IsAnyApplicationRunning() NN_NOEXCEPT
        {
            auto pid = GetRunningApplicationProcessId();
            return static_cast<bool>(pid);
        }
        Result RequestLinkDevice(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, const account::Uid& uid) NN_NOEXCEPT
        {
            return m_ECommerceManager.RequestLinkDevice(outHandle, outAsync, uid);
        }
        Result RequestSyncRights(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ns::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
        {
            return m_ECommerceManager.RequestSyncRights(outHandle, outAsync);
        }

        // ns_ApplicationDownloadApi.h
        Result RequestApplicationUpdateInfo(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.RequestApplicationUpdateInfo(outHandle, outAsync, id);
        }
        Result RequestUpdateApplication2(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.RequestUpdateApplication(outHandle, outAsync, id);
        }
        Result CancelApplicationDownload(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.CancelTask(id, TaskType::Download);
        }
        Result ResumeApplicationDownload(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.ResumeTask(id, TaskType::Download);
        }
        Result CalculateApplicationDownloadRequiredSize(sf::Out<ncm::StorageId> outStorageId, sf::Out<std::int64_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.CalculateTaskRequiredSize(outStorageId, outValue, id, TaskType::Download);
        }

        Result CancelApplicationApplyDelta(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.CancelTask(id, TaskType::ApplyDelta);
        }
        Result ResumeApplicationApplyDelta(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.ResumeTask(id, TaskType::ApplyDelta);
        }
        Result CalculateApplicationApplyDeltaRequiredSize(sf::Out<ncm::StorageId> outStorageId, sf::Out<std::int64_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_DownloadManager.CalculateTaskRequiredSize(outStorageId, outValue, id, TaskType::ApplyDelta);
        }
        Result ResumeAll() NN_NOEXCEPT
        {
            return m_DownloadManager.ResumeAll();
        }

        // ns_Ticket[System]Api.h
        Result IsActiveAccount(sf::Out<bool> outValue, es::AccountId accountId)
        {
            bool isActive = true;
            NN_RESULT_TRY(m_TicketManager.CheckAccountRights(accountId))
                NN_RESULT_CATCH(ResultApplicationLaunchRightsNotFound)
                {
                    isActive = false;
                }
            NN_RESULT_END_TRY

            *outValue = isActive;
            NN_RESULT_SUCCESS;
        }
        Result RequestDownloadApplicationPrepurchasedRights(sf::Out<sf::NativeHandle> outHandle, sf::Out<nn::sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_TicketManager.RequestDownloadApplicationPrepurchasedRights(outHandle, outAsync, id);
        }

        Result GetApplicationTicketInfo(nn::sf::Out<ns::ApplicationTicketInfo> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_TicketManager.GetApplicationTicketInfo(outValue, id);
        }

        // ns_ApplicationApiForRid.h
        Result GetApplicationDownloadTaskStatus(sf::Out<ApplicationDownloadTaskStatus> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ViewManager.GetApplicationDownloadTaskStatus(outValue, id);
        }
        Result DownloadApplication(ncm::ApplicationId applicationId, ncm::StorageId storageId, bool forceDirectUpdate) NN_NOEXCEPT
        {
            sf::NativeHandle nativeHandle;
            sf::SharedPointer<ns::detail::IAsyncResult> sp;
            NN_RESULT_DO(m_DownloadManager.RequestDownloadApplication(&nativeHandle, &sp, applicationId, storageId, forceDirectUpdate));
            return sp->Get();
        }
        Result RequestDownloadApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId applicationId, ncm::StorageId storageId, bool forceDirectUpdate) NN_NOEXCEPT
        {
            return m_DownloadManager.RequestDownloadApplication(outHandle, outAsync, applicationId, storageId, forceDirectUpdate);
        }
        Result RequestDownloadAddOnContent(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id, sf::InArray<ncm::ContentMetaKey> keyList, ncm::StorageId storageId) NN_NOEXCEPT
        {
            return m_DownloadManager.RequestDownloadAddOnContent(outHandle, outAsync, id, keyList, storageId);
        }
        Result DeleteApplicationContentEntity(const ncm::ContentMetaKey& key) NN_NOEXCEPT
        {
            return m_ApplicationEntityManager.DeleteApplicationContentEntity(key);
        }
        Result CreateUserAccount(account::Nickname name, sf::InBuffer image) NN_NOEXCEPT
        {
            return m_AccountProxyManager.CreateUserAccount(name, image);
        }

        // ns_MigrationApi.h
        Result GetLaunchRequiredVersion(nn::sf::Out<std::uint32_t> outValue, nn::ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.GetLaunchRequiredVersion(outValue, id);
        }

        Result UpgradeLaunchRequiredVersion(nn::ncm::ApplicationId id, std::uint32_t launchRequiredVersion) NN_NOEXCEPT
        {
            return m_ApplicationVersionManager.UpgradeLaunchRequiredVersion(id, launchRequiredVersion);
        }

        // ns_ContentDeliveryApi.h
        Result GetSystemDeliveryInfo(sf::Out<SystemDeliveryInfo> outValue) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.GetSystemDeliveryInfo(outValue.GetPointer());
        }
        Result CompareSystemDeliveryInfo(sf::Out<int> outValue, const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.CompareSystemDeliveryInfo(outValue.GetPointer(), lhs, rhs);
        }
        Result SelectLatestSystemDeliveryInfo(sf::Out<int> outValue, sf::InArray<SystemDeliveryInfo> list, const SystemDeliveryInfo& receiverSystemInfo, sf::InArray<ApplicationDeliveryInfo> receiverAppList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.SelectLatestSystemDeliveryInfo(outValue.GetPointer(), list.GetData(), static_cast<int>(list.GetLength()), receiverSystemInfo, receiverAppList.GetData(), static_cast<int>(receiverAppList.GetLength()));
        }
        Result VerifyDeliveryProtocolVersion(const SystemDeliveryInfo& info) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.VerifyDeliveryProtocolVersion(info);
        }
        Result GetApplicationDeliveryInfo(sf::Out<int> outCount, sf::OutArray<ApplicationDeliveryInfo> outList, ncm::ApplicationId id, ApplicationDeliveryAttribute attributes) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.GetApplicationDeliveryInfo(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), id, attributes);
        }
        Result HasAllContentsToDeliver(sf::Out<bool> outValue, sf::InArray<ApplicationDeliveryInfo> appList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.HasAllContentsToDeliver(outValue.GetPointer(), appList.GetData(), static_cast<int>(appList.GetLength()));
        }
        Result CompareApplicationDeliveryInfo(sf::Out<int> outValue, sf::InArray<ApplicationDeliveryInfo> lhs, sf::InArray<ApplicationDeliveryInfo> rhs) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.CompareApplicationDeliveryInfo(outValue.GetPointer(), lhs.GetData(), static_cast<int>(lhs.GetLength()), rhs.GetData(), static_cast<int>(rhs.GetLength()));
        }
        Result CanDeliverApplication(sf::Out<bool> outValue, sf::InArray<ApplicationDeliveryInfo> receiverInfoList, sf::InArray<ApplicationDeliveryInfo> senderInfoList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.CanDeliverApplication(outValue.GetPointer(), receiverInfoList.GetData(), static_cast<int>(receiverInfoList.GetLength()), senderInfoList.GetData(), static_cast<int>(senderInfoList.GetLength()));
        }
        Result ListContentMetaKeyToDeliverApplication(sf::Out<int> outCount, sf::OutArray<ncm::ContentMetaKey> outList, int offset, sf::InArray<ApplicationDeliveryInfo> infoList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.ListContentMetaKeyToDeliverApplication(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset, infoList.GetData(), static_cast<int>(infoList.GetLength()));
        }
        Result NeedsSystemUpdateToDeliverApplication(sf::Out<bool> outValue, sf::InArray<ApplicationDeliveryInfo> applicationList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.NeedsSystemUpdateToDeliverApplication(outValue.GetPointer(), applicationList.GetData(), static_cast<int>(applicationList.GetLength()), receiverSystemInfo);
        }
        Result EstimateRequiredSize(sf::Out<int64_t> outValue, sf::InArray<ncm::ContentMetaKey> keyList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.EstimateRequiredSize(outValue.GetPointer(), keyList.GetData(), static_cast<int>(keyList.GetLength()));
        }
        Result RequestReceiveApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id, ncm::StorageId storageId) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.RequestReceiveApplication(outHandle, outAsync, ipv4, port, keyList, id, storageId);
        }
        Result CommitReceiveApplication(ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.CommitReceiveApplication(id);
        }
        Result GetReceiveApplicationProgress(sf::Out<ReceiveApplicationProgress> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.GetReceiveApplicationProgress(outValue.GetPointer(), id);
        }
        Result RequestSendApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.RequestSendApplication(outHandle, outAsync, ipv4, port, keyList, id);
        }
        Result GetSendApplicationProgress(sf::Out<SendApplicationProgress> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.GetSendApplicationProgress(outValue.GetPointer(), id);
        }
        Result ListNotCommittedContentMeta(sf::Out<int> outCount, sf::OutArray<ncm::ContentMetaKey> outList, ncm::ApplicationId id, int offset) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.ListNotCommittedContentMeta(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), id, offset);
        }
        Result CreateDownloadTask(sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.CreateDownloadTask(keyList.GetData(), static_cast<int>(keyList.GetLength()), id);
        }
        Result GetApplicationDeliveryInfoHash(sf::Out<ApplicationDeliveryInfoHash> outValue, const sf::InArray<ApplicationDeliveryInfo> infoList) NN_NOEXCEPT
        {
            return m_ContentDeliveryManager.GetApplicationDeliveryInfoHash(outValue.GetPointer(), infoList.GetData(), static_cast<int>(infoList.GetLength()));
        }

        // ns_AppletLauncherApi.h
        Result GetApplicationShellEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
        {
            return m_AppletLauncher.GetApplicationShellEvent(outValue);
        }
        Result PopApplicationShellEventInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationShellEventInfo>& outList) NN_NOEXCEPT
        {
            return m_AppletLauncher.PopApplicationShellEventInfo(outCount, outList);
        }
        Result LaunchLibraryApplet(sf::Out<os::ProcessId> outValue, ncm::SystemProgramId id) NN_NOEXCEPT
        {
            return m_AppletLauncher.LaunchLibraryApplet(outValue, id);
        }
        Result TerminateLibraryApplet(os::ProcessId pid) NN_NOEXCEPT
        {
            return m_AppletLauncher.TerminateLibraryApplet(pid);
        }
        Result LaunchSystemApplet(sf::Out<os::ProcessId> outValue) NN_NOEXCEPT
        {
            return m_AppletLauncher.LaunchSystemApplet(outValue);
        }
        Result TerminateSystemApplet(os::ProcessId pid) NN_NOEXCEPT
        {
            return m_AppletLauncher.TerminateSystemApplet(pid);
        }
        Result LaunchOverlayApplet(sf::Out<os::ProcessId> outValue) NN_NOEXCEPT
        {
            return m_AppletLauncher.LaunchOverlayApplet(outValue);
        }
        Result TerminateOverlayApplet(os::ProcessId pid) NN_NOEXCEPT
        {
            return m_AppletLauncher.TerminateOverlayApplet(pid);
        }

        // ns_DynamicRightsApi.h
        Result RequestApplicationRightsOnServer(
            sf::Out<sf::NativeHandle> outHandle,
            sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync,
            ncm::ApplicationId id,
            const account::Uid& uid,
            Bit32 flags) NN_NOEXCEPT
        {
            return m_DynamicRightsManager.RequestApplicationRightsOnServer(outHandle, outAsync, id, uid, flags);
        }

        Result RequestAssignRights(
            sf::Out<sf::NativeHandle> outHandle,
            sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync,
            sf::InArray<ApplicationRightsOnServer> list) NN_NOEXCEPT
        {
            return m_DynamicRightsManager.RequestAssignRights(outHandle, outAsync, list);
        }

        Result RequestAssignRightsToResume(
            sf::Out<sf::NativeHandle> outHandle,
            sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync,
            RightsEnvironmentHandle handle,
            const account::Uid& uid) NN_NOEXCEPT
        {
            return m_DynamicRightsManager.RequestAssignRightsToResume(outHandle, outAsync, handle, uid);
        }

        Result RegisterUserOfAccountRestrictedRights(const account::Uid& uid) NN_NOEXCEPT
        {
            return m_DynamicRightsManager.RegisterUserOfAccountRestrictedRights(uid);
        }

        Result UnregisterAllUsersOfAccountRestrictedRights() NN_NOEXCEPT
        {
            m_DynamicRightsManager.UnregisterAllUsersOfAccountRestrictedRights();
            NN_RESULT_SUCCESS;
        }

        Result VerifyActivatedRightsOwners(RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            return m_DynamicRightsManager.VerifyActivatedRightsOwners(handle);
        }

        Result GetApplicationRightsStatus(sf::Out<ApplicationRightsStatus> outValue, RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            *outValue = m_RightsEnvironmentManager.GetApplicationRightsStatus(handle);
            NN_RESULT_SUCCESS;
        }

        // ns_PreInstallApi.h
        Result RequestCleanupAllPreInstalledApplications(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
        {
            return m_PreInstallManager.RequestCleanupAllPreInstalledApplications(outHandle, outAsync);
        }
        Result RequestCleanupPreInstalledApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_PreInstallManager.RequestCleanupPreInstalledApplication(outHandle, outAsync, id);
        }

        // ns_ApplicationRightsApi.h
        Result GetApplicationRightsOnClient(
            sf::Out<int> outCount,
            sf::OutArray<ApplicationRightsOnClient> outList,
            ncm::ApplicationId appId,
            const account::Uid& uid,
            ApplicationRightsOnClientQueryFlag flags) NN_NOEXCEPT
        {
            return m_ApplicationLaunchManager.GetApplicationRightsOnClient(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), appId, uid, flags);
        }

    public:
        // ns_ApplicationLaunchManager.h
        virtual Result AfterLaunchProgram(os::ProcessId processId, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationLaunchManager.AfterLaunchProgram(processId, info, programIndex);
        }
        virtual Result LaunchApplicationForDevelop(os::ProcessId* processId, const ApplicationLaunchInfo& info) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationLaunchManager.LaunchApplicationForDevelop(processId, info);
        }
        virtual Result GetApplicationLaunchInfoForDevelop(ApplicationLaunchInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationLaunchManager.GetApplicationLaunchInfo(outValue, id);
        }
        virtual Result RedirectApplicationProgramForDevelop(ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationLaunchManager.RedirectApplicationProgramForDevelop(id, programIndex);
        }

    public:

        Result CreateRightsEnvironment(sf::Out<RightsEnvironmentHandle> pOut) NN_NOEXCEPT
        {
            return m_RightsEnvironmentManager.CreateRightsEnvironment(pOut.GetPointer());
        }

        Result DestroyRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            m_RightsEnvironmentManager.DestroyRightsEnvironment(handle);
            NN_RESULT_SUCCESS;
        }

        Result AddTargetApplicationToRightsEnvironment(RightsEnvironmentHandle handle, ncm::ApplicationId applicationId) NN_NOEXCEPT
        {
            return m_RightsEnvironmentManager.AddTargetApplicationToRightsEnvironment(handle, applicationId, &m_Integrated, &m_RecordDb);
        }

        Result SetUsersToRightsEnvironment(RightsEnvironmentHandle handle, bool anyUser, util::Span<const account::Uid> uids) NN_NOEXCEPT
        {
            return m_RightsEnvironmentManager.SetUsersToRightsEnvironment(handle, anyUser, uids);
        }

        Result ActivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            return m_RightsEnvironmentManager.ActivateRightsEnvironment(handle);
        }

        Result DeactivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            m_RightsEnvironmentManager.DeactivateRightsEnvironment(handle);
            NN_RESULT_SUCCESS;
        }

        Result GetRightsEnvironmentStatus(sf::Out<RightsEnvironmentStatus> pOut, RightsEnvironmentHandle handle) NN_NOEXCEPT
        {
            return m_RightsEnvironmentManager.GetRightsEnvironmentStatus(pOut.GetPointer(), handle);
        }

        // ns_ApplicationCopyIdentifierApi.h
        Result GetGameCardApplicationCopyIdentifier(sf::Out<ns::GameCardApplicationCopyIdentifier> outValue, ncm::ApplicationId id) NN_NOEXCEPT
        {
            return m_ApplicationCopyIdentifierManager.GetGameCardApplicationCopyIdentifier(outValue, id);
        }
        Result GetInstalledApplicationCopyIdentifier(sf::Out<ns::InstalledApplicationCopyIdentifier> outValue, ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
        {
            return m_ApplicationCopyIdentifierManager.GetInstalledApplicationCopyIdentifier(outValue, id, storage);
        }

    private:
        pm::BootMode m_BootMode;

        ncm::ContentMetaDatabase m_BuildInUserDb;
        ncm::ContentStorage m_BuildInUserStorage;

        ncm::ContentMetaDatabase m_SdCardDb;
        ncm::ContentStorage m_SdCardStorage;

        AppletLauncher m_AppletLauncher;

        SystemReportManager* m_SystemReportManager;

        IntegratedContentManager m_Integrated;
        ApplicationRecordDatabase m_RecordDb;
        ApplicationControlDataManager m_ControlDataManager;

        DataStore m_DataStore;
        ShutdownManager m_ShutdownManager;

        ApplicationVersionManager m_ApplicationVersionManager;
        ApplicationViewManager m_ViewManager;

        ApplicationInstallRequestList m_RequestList;
        RequestServer m_RequestServer;
        DownloadTaskListManager m_DownloadTaskListManager;

        GameCardManager m_GameCardManager;
        SdCardManager m_SdCardManager;

        PushNotificationDispatcher m_PushNotificationDispatcher;
        GameCardRegistrationManager m_GameCardRegistrationManger;

        ApplicationVerificationManager m_ApplicationVerificationManager;
        VulnerabilityManager m_VulnerabilityManager;

        ApplicationEntityManager m_ApplicationEntityManager;
        ApplicationLaunchManager m_ApplicationLaunchManager;

        UserResourceManagerImpl m_UserResourceManager;
        FactoryResetManager m_FactoryResetManager;

        PseudoDeviceIdManager m_PseudoDeviceIdManager;

        RequestServerManager m_RequestServerManager;

        ApplicationDownloadManager m_DownloadManager;

        TicketManager m_TicketManager;

        AccountProxyManager m_AccountProxyManager;

        CommitManager m_CommitManager;

        ContentDeliveryManager m_ContentDeliveryManager;

        ECommerceManager m_ECommerceManager;

        BlackListManager m_BlackListManager;

        LockedBufferManager m_LockedBufferManager;

        PreInstallManager m_PreInstallManager;

        RightsEnvironmentManager m_RightsEnvironmentManager;

        ApplicationCopyIdentifierManager m_ApplicationCopyIdentifierManager;

        // LockedBuffer 用のバッファ
        // 利用者のうち、一番サイズが大きいものに合わせる
        static constexpr size_t LockedBufferSize = sizeof(ncm::StorageContentMetaKey) * 2048;
        Bit8 m_LockedBufferPool[LockedBufferSize];

        DynamicRightsManager m_DynamicRightsManager;

        bool m_IsContentsDeliveryDebugApiEnabled;

        static const int MaxOverlayQueueCount = 64;
        ovln::SenderForOverlayType m_Sender;

        bool IsMaintenanceMode() NN_NOEXCEPT
        {
            return m_BootMode == pm::BootMode_Maintenance;
        }

        bool IsSafeMode() NN_NOEXCEPT
        {
            return m_BootMode == pm::BootMode_SafeMode;
        }

        Result GetSystemUpdateContentPath(ncm::Path* outValue) NN_NOEXCEPT;

        Result InitializeBuildInContentManager() NN_NOEXCEPT;

        Result GetContentPath(lr::Path* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex) NN_NOEXCEPT;
        Result RegisterHtmlDocumentPatch(const ncm::ContentMetaKey& key, ncm::ProgramId id, uint8_t programIndex) NN_NOEXCEPT;

        bool IsContentsDeliveryDebugApiEnabled() NN_NOEXCEPT
        {
            return m_IsContentsDeliveryDebugApiEnabled;
        }
    };

}}}
