﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
* @file
* @brief    ローカルコンテンツ配信用 API に関するヘッダファイルです。
*
* @detals   このヘッダファイルに書かれている API はローカルコンテンツ配信を行う際に利用する API です。
*           ローカルコンテンツ配信の作業中はインフラを用いたタスクを止める必要があります。
*           インフラを用いたタスクを止めておくために、 nn::ns::GetRequestServerStopper() で得られた nn::ns::RequestServerStopper を
*           ローカルコンテンツ配信の作業中は保持し続けてください。
*/
#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ns/ns_ApplicationDeliveryInfo.h>
#include <nn/ns/ns_Async.h>
#include <nn/ns/ns_ContentDeliveryProgress.h>
#include <nn/ns/ns_SystemDeliveryInfo.h>

namespace nn { namespace ns {
    /*
    * @brief    システム配信情報を取得します。
    *
    * @return   処理の結果が返ります。
    */
    Result GetSystemDeliveryInfo(SystemDeliveryInfo* outValue) NN_NOEXCEPT;

    /*
    * @brief    nn::ns::SystemDeliveryInfo を比較します。
    *
    * @details  本体更新情報を比較します。
    *           outValue < 0: rhs の方が新しいです。
    *           outValue > 0: lhs の方が新しいです。
    *           outValue == 0: 同じバージョンです。

    * @return   処理の結果が返ります。
    * @retval   ResultInvalidSystemDeliveryInfo                 不正な SystemDeliveryInfo が渡されました。
    * @retval   ResultInvalidSystemDeliveryProtocolVersion      システム配信プロトコルのバージョンに対応していません
    */
    Result CompareSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT;

    /*
    * @brief    最新の本体更新情報を選択します。
    *
    * @details  list で与えられたシステム情報の中から、
    *           receiverSystemInfo, receiverAppInfoList の要求を満たす最新のシステムを選択します。
    *           list には nn::ns::VerifyDeliveryProtocolVersion() を満たすものを入れてください。
    *           outValue にはその選択されたシステム　が存在する list のインデックスが入れられます。
    *           適切なシステムをリストから見つけられなかった場合、outValue に -1 が入れられます。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidSystemDeliveryInfo                 不正な SystemDeliveryInfo が渡されました。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidSystemDeliveryProtocolVersion      システム配信プロトコルのバージョンに対応していません
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    */
    Result SelectLatestSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo& receiverSystemInfo, const ApplicationDeliveryInfo receiverAppInfoList[], int numAppInfoList) NN_NOEXCEPT;

    /*
    * @brief    最新の本体更新情報を選択します。
    *
    * @details  list で与えられたシステム情報の中から、最新のシステムを選択します。
    *           list には nn::ns::VerifyDeliveryProtocolVersion() を満たすものを入れてください。
    *           outValue にはその選択されたシステム　が存在する list のインデックスが入れられます。
    *           適切なシステムをリストから見つけられなかった場合、outValue に -1 が入れられます。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidSystemDeliveryInfo                 不正な SystemDeliveryInfo が渡されました。
    * @retval   ResultInvalidSystemDeliveryProtocolVersion      システム配信プロトコルのバージョンに対応していません
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    */
    Result SelectLatestSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT;

    /*
    * @brief    SystemDeliveryInfo のプロトコルバージョンを確認します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidSystemDeliveryInfo                 不正な SystemDeliveryInfo が渡されました。
    * @retval   ResultInvalidSystemDeliveryProtocolVersion      システム配信のプロトコルに対応していません。
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultSystemUpdateRequiredForLocalContentDelivery   ローカルコンテンツ配信必須システムバージョンを満たしていません
    *
    */
    Result VerifyDeliveryProtocolVersion(const SystemDeliveryInfo &info) NN_NOEXCEPT;

    /*
    * @brief    アプリケーション配信情報を取得します。
    *
    * @details  id で指定されるアプリケーションの配信情報を取得します。
    *           attributes にはセッションで取り扱うコンテンツの種別を指定します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound                 アプリケーション記録が見つかりません
    * @retval   ResultInvalidContentDeliveryRequest             attributes に指定された要求が不正です。
    * @retval   ResultBufferNotEnough                           numList が小さすぎます
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultApplyDeltaTaskExists                      適用中のタスクがあります
    */
    Result GetApplicationDeliveryInfo(int* outCount, ApplicationDeliveryInfo* outList, int numList, ncm::ApplicationId id, ApplicationDeliveryAttribute attributes) NN_NOEXCEPT;

    /*
    * @brief    配信可能状態かを判定します。
    *
    * @details  引数で与えられる nn::ns::ApplicationDeliveryInfo と nn::ns::RelativeApplicationDeliveryInfo を見て、
    *           配信するために必要なコンテンツを持っているかを確認します。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    */
    Result HasAllContentsToDeliver(bool* outValue, const ApplicationDeliveryInfo appInfoList[], int numList) NN_NOEXCEPT;

    /*
    * @brief    nn::ns::ApplicationDeliveryInfo を比較します。
    *
    * @details  配信するコンテンツのバージョンを比較します。
    *           ApplicationDeliveryInfo には、 nn::ns::HasAllContentsToDeliver() が true のものを入れてください。
    *           outValue < 0: rhs の方が新しいです。
    *           outValue > 0: lhs の方が新しいです。
    *           outValue == 0: 同じバージョンです。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    */
    Result CompareApplicationDeliveryInfo(int* outValue, const ApplicationDeliveryInfo lhsList[], int numLhsList, const ApplicationDeliveryInfo rhsList[], int numRhsList) NN_NOEXCEPT;

    /*
    * @brief    受信者に対して送信者が必要なコンテンツを持っているかを持っているかを調べます
    *
    * @details  nn::ns::ApplicationDeliveryInfo で要求されるコンテンツは送信者と受信者で一致している必要があります。
    *           要求するコンテンツを一致させるためには、nn::ns::GetApplicationDeliveryInfo() に与える attributes を一致させる必要があります。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    * @retval   ResultContentEntityNotFoundForContentDelivery   配信するコンテンツの実体が存在しません
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultApplicationUpdateRequired                 アプリケーションの更新が必要です。
    */
    Result CanDeliverApplication(bool* outValue, const ApplicationDeliveryInfo receiverInfoList[], int numReceiverInfoList, const ApplicationDeliveryInfo senderInfoList[], int numSenderInfoList) NN_NOEXCEPT;

    /*
    * @brief    送信者が必要なコンテンツを持っているかを調べます
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    * @retval   ResultContentEntityNotFoundForContentDelivery   配信するコンテンツの実体が存在しません
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultApplicationUpdateRequired                 アプリケーションの更新が必要です。
    */
    Result CanDeliverApplication(const ApplicationDeliveryInfo senderInfoList[], int numSenderInfoList) NN_NOEXCEPT;

    /*
    * @brief    配信する nn::ncm::ContentMetaKey のリストを取得します。
    *
    * @details  nn::ns::ApplicationDeliveryInfo と nn::ns::RelativeApplicationDeliveryInfo で指定される
    *           配信するコンテンツの ContentMetaKey を取得します。
    *           infoList で指定される ApplicationDeliveryInfo は nn::ns::HasAllContentsToDeliver() が True のものである必要があります。
    *           この API は ローカルコンテンツ配信の配信者が呼ぶ想定です。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    * @retval   ResultContentEntityNotFoundForContentDeliver    配信するコンテンツの実体が存在しません
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultApplyDeltaTaskExists                      適用中のタスクがあります
    */
    Result ListContentMetaKeyToDeliverApplication(int* outCount, ncm::ContentMetaKey outList[], int numList, int offset, const ApplicationDeliveryInfo infoList[], int numInfoList) NN_NOEXCEPT;

    /*
    * @brief    アプリケーション配信情報とシステム配信情報を比較して、本体更新が必要かを判断します。
    *
    * @details  appList で指定される ApplicationDeliveryInfo は nn::ns::HasAllContentsToDeliver() が True のものである必要があります。
    *           また、 receiverSystemInfo は nn::ns::VerifyDeliveryProtocolVersion() を満たしている必要があります。
    *           この API は ローカルコンテンツ配信のコーディネータが呼ぶ想定です。
    *
    * @return   処理の結果が返ります
    * @retval   ResultInvalidSystemDeliveryInfo                 不正な SystemDeliveryInfo が渡されました。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidSystemDeliveryProtocolVersion      受信側の SystemDeliveryInfo が読み取れません。
    * @retval   ResultInvalidContentDeliveryRequest             ApplicationDeliveryInfo に不正な要求が入っています
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    */
    Result NeedsSystemUpdateToDeliverApplication(bool* outValue, const ApplicationDeliveryInfo appList[], int numAppList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT;

    /*
    * @brief    配信先がコンテンツを受信するために必要とするストレージの空き容量を見積もります。
    *
    * @return   処理の結果が返ります
    */
    Result EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey keyList[], int numKey) NN_NOEXCEPT;

    /*
    * @brief    ローカルコンテンツ配信を用いたアプリケーションのコンテンツの受信を非同期処理で要求します。
    *
    * @return   処理の結果が返ります
    * @retval   ResultOutOfMaxRunningTask                   同時に起動できるタスクの上限数を超えています。
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    * @retval   ResultInvalidContentDeliveryRequest         引数に不正な要求が入っています。
    */
    Result RequestReceiveApplication(AsyncResult* outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId id, const ncm::ContentMetaKey keyList[], int numList, ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;

    /*
    * @brief    受信したアプリケーションのコンテンツを適用します。
    *
    * @return   処理の結果が返ります
    * @retval   ResultReceiveApplicationTaskNotFound        指定されたアプリケーションのタスクが見つかりません
    * @retval   ResultApplicationContentNotDownloaded       アプリケーションのコンテンツがまだダウンロードされていません
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    */
    Result CommitReceiveApplication(ncm::ApplicationId id) NN_NOEXCEPT;

    /*
    * @brief    アプリケーション受信タスクの進捗を取得します。
    *
    * @return   処理の結果が返ります
    * @retval   ResultReceiveApplicationTaskNotFound        指定されたアプリケーションのタスクが見つかりません
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    */
    Result GetReceiveApplicationProgress(ReceiveApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /*
    * @brief    ローカルコンテンツ配信を用いたアプリケーションのコンテンツの送信を非同期処理で要求します。
    *
    * @return   処理の結果が返ります
    * @retval   ResultOutOfMaxRunningTask                   同時に起動できるタスクの上限数を超えています。
    * @retval   ResultApplicationRecordNotFound             アプリケーション記録がありません
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    * @retval   ResultInvalidContentDeliveryRequest         引数に不正な要求が入っています。
    */
    Result RequestSendApplication(AsyncResult* outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId id, const ncm::ContentMetaKey keyList[], int numList) NN_NOEXCEPT;

    /*
    * @brief    アプリケーションの送信タスクの進捗を取得します。
    *
    * @return   処理の結果が返ります
    * @retval   ResultSendApplicationTaskNotFound           指定されたアプリケーションのタスクが見つかりません。
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    */
    Result GetSendApplicationProgress(SendApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /*
    * @brief    現在まだコミットされていないコンテンツメタのリストを取得します。
    *
    * @pre
    *           - 適用タスクが存在しない
    *
    * @return   リストされたコンテンツの数を返します。
    * @retval   ResultApplyDeltaTaskExists                  適用中のタスクがあります
    */
    int ListNotCommittedContentMeta(ncm::ContentMetaKey outList[], int listCount, ncm::ApplicationId id, int offset) NN_NOEXCEPT;

    /*
    * @brief    ダウンロードタスクを生成します。
    *
    * @details  nn::ns::ListUncommittedContentMeta() で取得したコンテンツメタのリストを利用して、ダウンロードタスクを作成します。
    *
    * @pre
    *           - 適用タスクが存在しない
    *           - 既存のダウンロードタスクが存在しない
    *           - listCount > 0
    */
    void CreateDownloadTask(const ncm::ContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT;

     /*
    * @brief    アプリケーション配信情報が更新されているかを調べるための情報を取得します

    * @details  GetApplicationDeliveryInfo() で取得した ApplicationDeliveryInfo を指定してください。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultInvalidApplicationDeliveryInfo            不正な ApplicationDeliveryInfo が渡されました。
    * @retval   ResultInvalidApplicationDeliveryProtocolVersion アプリケーション配信のプロトコルに対応していません
    * @retval   ResultBufferNotEnough                           作業用バッファが確保できませんでした。
    */
    Result GetApplicationDeliveryInfoHash(ApplicationDeliveryInfoHash* outValue, const ApplicationDeliveryInfo infoList[], int numList) NN_NOEXCEPT;

}}  // namespace ns
