﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/kvdb/kvdb_BoundedString.h>

namespace nn { namespace nim { namespace srv {

    class TaskDataStore;
    class IdIterator
    {
        friend TaskDataStore;

    public:
        util::optional<util::Uuid> Next() NN_NOEXCEPT;

    private:
        util::Uuid* m_List{};
        int m_Count{};
        int m_MaxCount{};
    };

    class TaskDataStore
    {
    public:
        typedef kvdb::BoundedString<64> Path;

        ~TaskDataStore() NN_NOEXCEPT;

        Result Initialize(const char* mountName, fs::SystemSaveDataId id, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags) NN_NOEXCEPT;

        Result Create(const util::Uuid& uuid, const void* meta, size_t metaSize) NN_NOEXCEPT;
        Result Destroy(const util::Uuid& uuid) NN_NOEXCEPT;
        Result GetMetaSize(size_t* outValue, const util::Uuid& uuid) NN_NOEXCEPT;
        Result GetMeta(size_t* outValue, const util::Uuid& uuid, void* buffer, size_t bufferSize) NN_NOEXCEPT;
        Result GetMetaFilePath(Path* outValue, const util::Uuid& uuid) NN_NOEXCEPT;
        Result GetDataFilePath(Path* outValue, const util::Uuid& uuid) NN_NOEXCEPT;

        IdIterator CreateIdIterator() NN_NOEXCEPT;

        Result Commit() NN_NOEXCEPT;

        // アンマウントしてセーブデータを破棄する
        // 破棄した後は再 Initialize が必要
        Result DestroySaveData() NN_NOEXCEPT;

        static Result DeleteSaveData(fs::SystemSaveDataId id) NN_NOEXCEPT;

    private:
        Path MakeRootDirecotryPath() NN_NOEXCEPT;
        Path MakeTaskDirectoryPath(const util::Uuid& uuid) NN_NOEXCEPT;
        Path MakeTaskDataFilePath(const util::Uuid& uuid) NN_NOEXCEPT;
        Path MakeTaskMetaFilePath(const util::Uuid& uuid) NN_NOEXCEPT;
        Path MakeTaskListFilePath() NN_NOEXCEPT;

        void ClearList() NN_NOEXCEPT;
        Result ReadList() NN_NOEXCEPT;
        Result WriteList() NN_NOEXCEPT;
        Result AddToList(const util::Uuid& uuid) NN_NOEXCEPT;
        Result RemoveFromList(const util::Uuid& uuid) NN_NOEXCEPT;
        int Count() NN_NOEXCEPT;
        bool HasSpace() NN_NOEXCEPT;

        util::optional<Path> m_MoundName;
        fs::SystemSaveDataId m_Id;

        static const int MaxTaskCount = 64;
        util::Uuid m_List[MaxTaskCount];
    };

}}}
