﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/fs/fs_File.h>

namespace nn { namespace nim { namespace srv {

    class FileNetworkInstallTaskMeta
    {
    public:
        static Result Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT;
        Result Initialize(const char* path) NN_NOEXCEPT;
        Result Add(const ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT;
        Result SetAttribute(Bit64 attribute) NN_NOEXCEPT;
        ~FileNetworkInstallTaskMeta() NN_NOEXCEPT;

        int CountContentMeta() NN_NOEXCEPT
        {
            return m_Header.contentMetaCount;
        }
        Result GetContentMeta(ncm::ContentMetaKey* outValue, int index) NN_NOEXCEPT;
        ncm::ApplicationId GetApplicationId() const NN_NOEXCEPT
        {
            return m_Header.applicationId;
        }
        ncm::StorageId GetInstallStorage() const NN_NOEXCEPT
        {
            return m_Header.storage;
        }
        Bit32 GetConfig() const NN_NOEXCEPT
        {
            return m_Header.config;
        }
        Bit64 GetAttribute() const NN_NOEXCEPT
        {
            return m_Header.attribute;
        }

    private:
        struct Header
        {
            ncm::ApplicationId applicationId;
            int contentMetaCount;
            ncm::StorageId storage;
            Bit8 reserved1[3];
            Bit32 config;
            Bit8 reserved2[4];
            Bit64 attribute;
            Bit8 reserved3[224];
        };
        NN_STATIC_ASSERT(sizeof(Header) == 256);

        static Header MakeHeader(ncm::ApplicationId appId, int contentMetaCount, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT
        {
            Header header = {};
            header.applicationId = appId;
            header.contentMetaCount = contentMetaCount;
            header.storage = storage;
            header.config = config;

            return header;
        }

        static int64_t CalculateSize(int contentMetaCount) NN_NOEXCEPT
        {
            return sizeof(Header) + sizeof(ncm::ContentMetaKey) * contentMetaCount;
        }

        Result HasId(bool* outValue, Bit64 id) NN_NOEXCEPT;

        util::optional<fs::FileHandle> m_File;
        Header m_Header;
    };
}}}
