﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/account/account_Types.h>
#include <nn/nim/nim_Async.h>
#include <nn/nim/nim_DynamicRightsTypes.h>
#include <nn/npns/npns_Types.h>

namespace nn { namespace nim {

typedef detail::AsyncValueList<AvailableELicense> AsyncAvailableELicenses;

/**
   @brief   このデバイスで、指定したユーザが利用可能な eLicense をサーバに問い合わせます。

   @param[in]   uid     NA連携済ユーザアカウントIDを指定します。@n
                        NA未連携ユーザアカウントを指定した場合、状態に応じたACCOUNTライブラリのエラー値が返されます。@n
                        @ref nn::account::ResultNetworkServiceAccountUnavailable など。

   @details サーバ上で割り当て可能、割り当て済みの eLicense 情報が返ります。
            Assignable なものがある場合、RequestAssignELicense を呼ぶことで実際の eLicense の割り当てを行います。
*/
Result RequestQueryAvailableELicenses(AsyncAvailableELicenses* outValue, const account::Uid& uid, const es::RightsId* rightsIds, int rightsCount) NN_NOEXCEPT;

/**
   @brief このデバイスで利用可能な eLicense をサーバに問い合わせます。

   @details こちらの uid を指定しない版では、機器認証の権利のみ参照できます。
*/
Result RequestQueryAvailableELicenses(AsyncAvailableELicenses* outValue, const es::RightsId* rightsIds, int rightsCount) NN_NOEXCEPT;

typedef detail::AsyncValueList<AssignedELicense> AsyncAssignedELicenses;

/**
   @brief eLicense の発行をサーバに要求します。

   @param[in]   uid     NA連携済ユーザアカウントIDを指定します。@n
                        NA未連携ユーザアカウントを指定した場合、状態に応じたACCOUNTライブラリのエラー値が返されます。@n
                        @ref nn::account::ResultNetworkServiceAccountUnavailable など。

   @details 指定した Rights ID のライセンスの割り当てを要求します。
            割り当てた eLicense の同期は自動では行いません。RequestSyncELicenses を明示的に呼んでください。
            @ref AsyncAssignedELicenses::GetSize() == 0 の場合、要求された @ref es::RightsId 群に割り当て可能な eLicense が存在しない事を示します。
*/
Result RequestAssignELicenses(AsyncAssignedELicenses* outValue, const account::Uid& uid, const es::RightsId* rightsIds, int rightsCount, ELicenseType elicenseType) NN_NOEXCEPT;

/**
   @brief eLicense の発行をサーバに要求します。

   @details こちらの uid を指定しない版では、機器認証の権利のみ割り当て要求できます。
            割り当てた eLicense の同期は自動では行いません。RequestSyncELicenses を明示的に呼んでください。
            @ref AsyncAssignedELicenses::GetSize() == 0 の場合、要求された @ref es::RightsId 群に割り当て可能な eLicense が存在しない事を示します。
*/
Result RequestAssignELicenses(AsyncAssignedELicenses* outValue, const es::RightsId* rightsIds, int rightsCount, ELicenseType elicenseType) NN_NOEXCEPT;

/**
   @brief eLicense の発行をサーバに要求します。( デバイスに機器認証している全 NintendoAccount が対象 )

   @details 呼び出したデバイスに対して機器認証している全 NintendoAccount の全所有権利に対するライセンスの割り当てを要求します。
            割り当て要求の結果には既に割り当て済のライセンスも含まれます。
            割り当てた eLicense の同期は自動では行いません。RequestSyncELicenses を明示的に呼んでください。
*/
Result RequestAssignAllDeviceLinkedELicenses(AsyncAssignedELicenses* outValue) NN_NOEXCEPT;

/**
   @brief eLicense の有効期間延長を要求します。

   @param[in]   uid     NA連携済ユーザアカウントIDを指定します。@n
                        NA未連携ユーザアカウントを指定した場合、状態に応じたACCOUNTライブラリのエラー値が返されます。@n
                        @ref nn::account::ResultNetworkServiceAccountUnavailable など。

   @details 指定した eLicense ID の eLicense の利用期限延長を行います。
            延長した eLicense の同期は自動では行いません。RequestSyncELicenses を明示的に呼んでください。

            延長に成功するには、すべての eLicense がサーバ上で有効である必要があります。

            要求した eLicenseId 群の全てが、@ref AsyncAssignedELicenses::Get() で取得した延長成功結果リストに含まれない場合は延長は失敗したとみなします。
            但し、失敗した場合も eLicense 同期 ( @ref RequestSyncELicenses() の呼び出し )を行ってください。
            サーバ上でのライセンス剥奪などの通知を受け取りそこなったケースなどのサーバー・クライアント間不整合を解決するため同期を推奨します。
 */
Result RequestExtendELicenses(AsyncAssignedELicenses* outValue, const account::Uid& uid, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT;

/**
   @brief 指定したユーザの eLicense をサーバと同期します。

   @details 同期した後の権利状態は本 API では取得できません。
            同期後、改めて es に権利情報を問い合わせてください。
*/
Result RequestSyncELicenses(AsyncResult* outValue, const account::NintendoAccountId& naId) NN_NOEXCEPT;

/**
   @brief 指定した eLicense に対応する eTicket を取得します。

   @param[in]   uid     NA連携済ユーザアカウントIDを指定します。@n
                        NA未連携ユーザアカウントを指定した場合、状態に応じたACCOUNTライブラリのエラー値が返されます。@n
                        @ref nn::account::ResultNetworkServiceAccountUnavailable など。

   @details uid には eLicense の所有者の uid を指定してください。
            機器認証で付与された eLicense に対応するチケットは、この API ではダウンロードできません。
            代わりに ec::system::RequestDownloadTicket() を利用してください。

   @note    本機能は要求数に応じて分割ダウンロードを実施する可能性があります。@n
            その場合 @ref AsyncResult::Cancel() の呼び出した場合も、一部チケットがダウンロード・インポート済になる可能性があります。
*/
Result RequestDownloadETickets(AsyncResult* outValue, const account::Uid& uid, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT;

typedef detail::AsyncValue<RevokeReason> AsyncRevokeReason;

/**
   @brief eLicense が無効化された理由を問い合わせます。

   @details naId には、eLicense の所有者を指定してください。

   @retval ResultELicenseNotFound  指定した eLicense ID が正しくありません。
 */
Result RequestQueryRevokeReason(AsyncRevokeReason* outValue, const account::NintendoAccountId& naId, const es::ELicenseId& eLicenseId) NN_NOEXCEPT;

/**
   @brief 現状の権利利用状況をサーバに送信することを要求します。

   @details naId には、eLicense の所有者を指定してください。
*/
Result RequestReportActiveELicenses(AsyncResult* outValue, const account::NintendoAccountId& naId, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT;

/**
   @brief 現状の権利利用状況をサーバに送信することを要求します。

   @details サーバから要求された場合には、こちらの API で利用状況を報告してください。
*/
Result RequestReportActiveELicensesPassively(AsyncResult* outValue, const account::NintendoAccountId& naId, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT;

/**
   @brief npns から取得した通知トークンをサーバに登録します。

   @details 既に通知トークンがサーバに登録されていた場合は新しい通知トークンで上書きされます。
*/
Result RequestRegisterDynamicRightsNotificationToken(AsyncResult* outValue, const npns::NotificationToken& token) NN_NOEXCEPT;

}}
