﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesNetwork.h>
#include <nn/util/util_Uuid.h>


namespace nn
{
namespace nifm
{

    /**
     * @brief   利用要求の内容のプリセットを表します。
     */
    enum RequirementPreset
    {
        RequirementPreset_Invalid,                             //!< 無効なプリセット

        RequirementPreset_None,                                //!< 空のプリセット

        RequirementPreset_InternetBestEffort,                  //!< （アプリ向け）一部 BG サービスのネットワーク利用を制限するインターネット接続用プリセット、アプリ向けデフォルト
        RequirementPreset_InternetGeneric,                     //!< （アプリ向け）BG サービスのネットワーク利用を制限しないインターネット接続用プリセット
        RequirementPreset_LocalGeneric,                        //!< （アプリ向け）標準的なローカル通信用プリセット (ldn 経由で利用)

        RequirementPreset_InternetForApplet,                   //!< （システムアプリ・アプレット向け）一般アプレット用のインターネット接続用プリセット
        RequirementPreset_InternetForNetConnectApplet,         //!< （システムアプリ・アプレット向け）ネットワーク接続アプレット専用プリセット
        RequirementPreset_InternetForWifiWebAuthApplet,        //!< （システムアプリ・アプレット向け）Web 認証アプレット専用プリセット
        RequirementPreset_LocalForApplet,                      //!< （システムアプリ・アプレット向け）一般アプレット用のローカル通信用プリセット (ldn 経由で利用)

        RequirementPreset_InternetForSystemProcess,            //!< （システムプロセス向け）BG サービス用の、単発かつネットワーク利用制限の影響を受けるインターネット接続用プリセット
        RequirementPreset_InternetForSystemProcessPersistent,  //!< （システムプロセス向け）BG サービス用の、永続的かつネットワーク利用制限の影響を受けるインターネット接続用プリセット
        RequirementPreset_InternetForSystemProcessContinuous,  //!< （システムプロセス向け）BG サービス用の、永続的かつネットワーク利用制限の影響を受けないインターネット接続用プリセット

        // ビルド済みバイナリの前方互換性を破壊しないように、リリース後の追加プリセットはここより下に順番に追加する

        RequirementPreset_InternetForSystemProcessSharable,      //!< （システムプロセス向け）BG サービス用の、単発かつネットワーク利用制限の影響を受けないインターネット接続用プリセット
        RequirementPreset_InternetForDevMenu,                    //!< （システムアプリ・アプレット向け）DevMenu 専用プリセット
        RequirementPreset_InternetForNpns,                       //!< （システムプロセス向け）通知サービス専用の、スリープ時にも維持されることを想定した接続用のプリセット
        RequirementPreset_InternetForAppletContinuous,           //!< （システムアプリ・アプレット向け）一般アプレット用の、永続的かつネットワーク利用制限の影響を受けないインターネット接続用プリセット
        RequirementPreset_InternetBestEffortForApplet,           //!< （システムアプリ・アプレット向け）一般アプレット用の、一部 BG サービスのネットワーク利用を制限するインターネット接続用プリセット
        RequirementPreset_InternetBestEffortForAppletPersistent, //!< （システムアプリ・アプレット向け）一般アプレット用の、永続的かつ一部 BG サービスのネットワーク利用を制限するインターネット接続用プリセット
        RequirementPreset_NeighborDetection,                     //!< （アプリ向け）一般アプリ用の、すれ違い通信向けプリセット
        RequirementPreset_NeighborDetectionForApplet,            //!< （システムアプリ・アプレット向け）一般アプレット用の、すれ違い通信向けプリセット
        RequirementPreset_NeighborDetectionForSystemProcess,     //!< （システムプロセス向け）BG サービス用の、すれ違い通信向けプリセット
        RequirementPreset_Scan,                                  //!< （アプリ向け）一般アプリ用の、パッシブスキャンプリセット
        RequirementPreset_ScanForApplet,                         //!< （システムアプリ・アプレット向け）システムアプレット用の、パッシブスキャンプリセット

        RequirementPreset_Count,

        RequirementPreset_Reserved1,
    };

    /**
     * @brief   利用要求構築時に受け渡すパラメータの型です。
     */
    struct RequestParameters
    {
        RequirementPreset requirementPreset;
    };

    /**
     * @brief   インターネット疎通確認や WISPr 認証に関する要求を示すオプションです。
     *
     * @details
     *  インターネットの利用要求以外では無視されます。
     */
    enum ConnectionConfirmationOption
    {
        ConnectionConfirmationOption_Invalid     = 0,   //!< 無効なオプションです

        ConnectionConfirmationOption_Prohibited  = 1,   //!< 疎通確認等をおこなわないように求めます
        ConnectionConfirmationOption_NotRequired = ConnectionConfirmationOption_Prohibited, //!< （互換性維持用）
        ConnectionConfirmationOption_Preferred   = 3,   //!< 疎通確認等をおこなうように求めますが、疎通が確認できなくても要求は受理されます
        ConnectionConfirmationOption_Required    = 4,   //!< 疎通確認等をおこなうように求めますが、すでに確認済みの接続に対しては再確認をおこないません
        ConnectionConfirmationOption_Forced      = 5,   //!< 接続の状態にかかわらず、要求評価の際にあらためて疎通確認等をおこなうように求めます

        ConnectionConfirmationOption_Count
    };

    /**
     * @brief   利用要求の内容を表します。
     *
     * @details
     *  この構造体は、内容の互換の維持は保証されません。
     */
    struct Requirement
    {
        uint8_t _priority;                                              //!< 優先度
        bool _isRejectable;                                             //!< 受理済み要求が却下されうるか否か
        bool _isPersistent;                                             //!< 一度受理されなかった場合にそのまま保留として残るか否か
        bool _isInstant;                                                //!< 提出した要求の即時評価を求めるか否か
        bool _isSustainable;                                            //!< 接続維持のために利用要求が内部的に複製されるか否か
        bool _isGreedy;                                                 //!<
        bool _isSharable;                                               //!<
        bool _isForeground;                                             //!< ユーザーに意識される要求か否か
        bool _isAutomaticSwitchProhibited;                              //!< 無線から有線への自動切換を抑制するか否か
        bool _isKeptInSleep;                                            //!< スリープ中も受理が維持されるか否か
        bool _isInternalContextUpdatable;                               //!< テレメトリコンテキストの更新に影響するか否か
        NetworkType _networkType;                                       //!< ネットワークの種類
        nn::util::Uuid _profileId;                                      //!< 接続設定 ID
        ConnectionConfirmationOption _connectionConfirmationOption;     //!< 疎通確認に関する要求
        // TODO: ETH
        // TODO: NIC ID
    };

}
}

