﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

namespace nn
{
namespace nifm
{
    /**
     * @brief リンク層の種類を表します。
     */
    enum NetworkInterfaceType
    {
        NetworkInterfaceType_Invalid,    //!< 無効
        NetworkInterfaceType_Ieee80211,  //!< IEEE 802.11
        NetworkInterfaceType_Ethernet,   //!< Ethernet

        NetworkInterfaceType_Count
    };

    /**
     * @brief 列挙時に指定するフィルタです。
     */
    enum EnumerateNetworkInterfacesFilter
    {
        EnumerateNetworkInterfacesFilter_Ieee80211  = 1 <<  0,  //!< IEEE 802.11 の無線通信インターフェースを列挙対象に含めます
        EnumerateNetworkInterfacesFilter_Ethernet   = 1 <<  1,  //!< Ethernet の有線通信インターフェースを列挙対象に含めます
        EnumerateNetworkInterfacesFilter_Extended   = 1 << 16,  //!< 利用対象でない NIC も列挙対象に含めます
    };

    /**
     * @brief MACアドレスを表す構造体です。
     */
    struct MacAddress
    {
        static const int8_t Size = 6;  //!< MAC アドレスを表すバイト列のサイズ
        uint8_t data[Size];            //!< MAC アドレス

        /**
         * @brief   等価比較演算子のオーバーロードです。
         *
         * @param[in]   lhs     比較対象の MacAddress を指定します。
         * @param[in]   rhs     比較対象の MacAddress を指定します。
         *
         * @return
         *  比較した両者が等しければ true を、等しくなければ false を返します。
         */
        friend bool operator==(const MacAddress& lhs, const MacAddress& rhs) NN_NOEXCEPT
        {
            return lhs.data[0] == rhs.data[0] &&
                   lhs.data[1] == rhs.data[1] &&
                   lhs.data[2] == rhs.data[2] &&
                   lhs.data[3] == rhs.data[3] &&
                   lhs.data[4] == rhs.data[4] &&
                   lhs.data[5] == rhs.data[5];
        }

        /**
         * @brief   非等価比較演算子のオーバーロードです。
         *
         * @param[in]   lhs     比較対象の MacAddress を指定します。
         * @param[in]   rhs     比較対象の MacAddress を指定します。
         *
         * @return
         *  比較した両者が等しくなければ true を、等しければ false を返します。
         */
        friend bool operator!=(const MacAddress& lhs, const MacAddress& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }
    };

    /**
     * @brief 取得可能なネットワークインタフェースの情報の最大数です。
     */
    static const int32_t NetworkInterfaceCountMax = 2;

    /**
     * @brief ネットワークインタフェース情報を表す構造体です。
     */
    struct NetworkInterfaceInfo
    {
        NetworkInterfaceType type;  //!< リンク層の種類
        MacAddress macAddress;      //!< MACアドレス
        bool isAvailable;           //!< 利用の対象となるか
    };
}
}

