﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>

#include <nn/util/util_TypedStorage.h>


namespace nn
{
namespace nifm
{

/**
 * @brief 一時的な接続設定を表します。
 */
class TemporaryNetworkProfile
{
    NN_DISALLOW_COPY(TemporaryNetworkProfile);
    NN_DISALLOW_MOVE(TemporaryNetworkProfile);

private:
    using Storage = nn::util::TypedStorage<char, 40U, sizeof(void*)>;

private:
    Storage m_Storage;

public:
    /**
     * @brief   一時的な接続設定を扱うためのオブジェクトを構築します。
     *
     * @param[in]   profileData     一時的にシステムへ登録する接続設定の内容を指定します。
     *
     * @pre     事前にネットワーク接続ライブラリが初期化されている必要があります。
     *
     * @details
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  TemporaryNetworkProfile クラスのオブジェクトを大域的に配置することはできません。
     */
    explicit TemporaryNetworkProfile(const NetworkProfileData& profileData) NN_NOEXCEPT;

    /**
     * @brief デストラクタです。
     *
     * @details
     *  オブジェクトを破棄した場合、接続設定のシステムへの登録も取り下げられます。
     *  取り下げられた接続設定をシステムのネットワーク接続が利用していた場合、接続が切断されることに注意してください。
     *  ネットワーク接続を維持したい間は、一時的な接続設定オブジェクトを維持する必要があります。
     */
    ~TemporaryNetworkProfile() NN_NOEXCEPT;

    /**
     * @brief   一時的な接続設定の ID を取得します。
     *
     * @return  一時的な接続設定の ID が返ります。
     *
     * @details
     *  一時的な接続設定の ID を取得します。
     */
    const nn::util::Uuid& GetId() const NN_NOEXCEPT;

    /**
     * @brief パラメータを変更してテンポラリ接続設定を再生成します。
     *
     * @param[in]   profileData     一時的にシステムへ登録する接続設定の内容を指定します。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  一時的な接続設定を再生成した場合、以前の内容の登録は取り下げられます。
     *  取り下げられた接続設定をシステムのネットワーク接続が利用していた場合、接続が切断されることに注意してください。
     *  ネットワーク接続を維持したい間は、接続設定の内容を維持する必要があります。@n
     *
     *  再生成後は ID も変更されます。
     */
    Result Update(const NetworkProfileData& profileData) NN_NOEXCEPT;

    /**
     * @brief テンポラリ接続設定のハンドルを取得します。
     *
     * @return テンポラリ接続設定のハンドルが返ります。
     *
     * @details
     */
    NetworkProfileHandle GetHandle() NN_NOEXCEPT;
};
}
}

