﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesScan.h>
#include <nn/nifm/nifm_TypesAccessPoint.h>


namespace nn
{
namespace nifm
{
    //! @name アプリケーション向けスキャン関連 API
    //! @{

    /**
     * @brief 明示的に周囲の無線環境をスキャンします。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - 異なるチャンネルを指定したスキャンの実行中ではない
     *
     * @details
     *  スキャンが完了するまでブロックします。
     *  使用可能なすべてのチャンネルに対してパッシブスキャンを行います。
     *
     *  アプリが提出済みのインターネット通信以外の利用要求を却下してからスキャンを行うため、必要に応じて利用者に確認ダイアログなどを表示してください。
     */
    nn::Result Scan() NN_NOEXCEPT;

    /**
     * @brief 明示的に指定されたチャンネルにおける周囲の無線環境をスキャンします。
     *
     * @param[in] scanChannels  スキャン対象となるチャンネルの配列
     * @param[in] count         ¥a scanChannels で指定されたチャンネルの個数
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - 異なるチャンネルを指定したスキャンの実行中ではない
     *
     * @details
     *  スキャンが完了するまでブロックします。
     *  指定されたチャンネルに対してパッシブスキャンを行います。
     *
     *  アプリが提出済みのインターネット通信以外の利用要求を却下してからスキャンを行うため、必要に応じて利用者に確認ダイアログなどを表示してください。
     *
     */
    nn::Result Scan(const int16_t scanChannels[], int count) NN_NOEXCEPT;

    /**
     * @brief 無線デバイスで使用の許可されたチャンネルリストを取得します。
     *
     * @param[out] pOutChannels  無線デバイスで使用の許可されたチャンネルリストが格納されるバッファ
     * @param[out] pOutCount     実際の使用許可チャンネル数
     * @param[in] bufferCount   チャンネル数で表される、用意されたバッファのサイズ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  出力用バッファ \a pOutChannels には、 \a bufferCount 以下の個数しか格納されません。
     *  すべての出力がバッファに格納できないような場合、 \a pOutCount には \a bufferCount よりも大きい値が入ります。@n
     *  使用許可チャンネル数の上限値は @ref nn::nifm::WirelessChannelsCountMax で定義されます。@n
     *
     *  取得されるチャンネルは降順にソートされています。
     */
    nn::Result GetAllowedChannels(int16_t* pOutChannels, int* pOutCount, int bufferCount) NN_NOEXCEPT;

    /**
     * @brief スキャンの結果得られるアクセスポイント情報一覧を取得します。
     *
     * @param[out] pOutAccessPointData アクセスポイント情報を格納するバッファへのポインタ
     * @param[out] pOutCount           実際に取得されたアクセスポイント情報の個数
     * @param[in]  bufferCount         アクセスポイント情報数で表される、用意されたバッファのサイズ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  アクセスポイント情報は、一種類 AP ビーコンにつき一つ得られ、電波強度の強い順に格納されます。
     *  出力用バッファ \a pOutAccessPointData には、 \a bufferCount 以下の個数しか格納されません。
     *  すべての出力がバッファに格納できないような場合、 \a pOutCount には \a bufferCount よりも大きい値が入ります。@n
     *  一種類の AP ビーコンが複数の認証・暗号化方式をサポートする場合、この API で取得できるのは
     *  （Wpa2Psk > WpaPsk > Shared > Open > Unknown、Aes > Wep > None）で最も大きい値を持つもののみです。@n
     *
     *  事前に nifm::Scan によってチャンネルが指定されていた場合も、全てのチャンネルに関するアクセスポイント情報が返り得ますが、
     *  指定されたチャンネルのもののみが直前のスキャンによって得られた情報（最新の情報）であることが保証されます。
     */
    nn::Result GetScanData(AccessPointData* pOutAccessPointData, int* pOutCount, int bufferCount) NN_NOEXCEPT;

    //! @}
}
}
