﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesRequest.h>

#include <nn/applet/applet_LibraryApplet.h>

namespace nn
{
namespace nifm
{
    //! @name 利用要求の結果に対するエラーハンドリング API
    //! @{

    /**
     * @brief 利用要求の結果に対するエラーハンドリングのための準備を行います。
     *
     * @param[in]  handle                     利用要求ハンドル
     * @param[out] pOutLibraryAppletHandle    ライブラリアプレットハンドル
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求が却下済み
     *   - pOutLibraryAppletHandle != nullptr
     *
     * @details
     *  この関数は Winding が必要なライブラリアプレットでエラーハンドリングを行うために利用します。
     *  それ以外の場合 HandleNetworkRequestResult() を利用してください。
     */
    nn::Result PrepareHandlingNetworkRequestResult(RequestHandle handle, nn::applet::LibraryAppletHandle* pOutLibraryAppletHandle) NN_NOEXCEPT;

    /**
     * @brief 利用要求の結果に対するエラーハンドリングを開始します。
     *
     * @param[in] libraryAppletHandle    ライブラリアプレットハンドル
     * @param[in] pUserArgument          内部で nn::applet::StartLibraryApplet に渡されるユーザー引数
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - libraryAppletHandle が有効な値
     *
     * @details
     *  この関数は Winding が必要なライブラリアプレットでエラーハンドリングを行うために利用します。
     *  それ以外の場合 HandleNetworkRequestResult() を利用してください。
     */
    nn::Result StartHandlingNetworkRequestResult(const nn::applet::LibraryAppletHandle& libraryAppletHandle, void* pUserArgument) NN_NOEXCEPT;

    /**
     * @brief 利用要求の結果に対するエラーハンドリングを終了します。
     *
     * @param[in] libraryAppletHandle    ライブラリアプレットハンドル
     *
     * @retresult
     *   @handleresult{
     *     ResultErrorHandlingCompleted, エラーハンドリングが正常に行われました。エラーが解消された可能性があります。
     *   }
     *   - その他の Result の場合、エラーハンドリングに失敗しています。
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - libraryAppletHandle が有効な値
     *
     * @details
     *  この関数は Winding が必要なライブラリアプレットでエラーハンドリングを行うために利用します。
     *  それ以外の場合 HandleNetworkRequestResult() を利用してください。
     *
     *  この関数は Unwound 後に呼び出します。
     *  この関数が
     *    - ResultErrorHandlingCompleted を返した場合、利用要求を再提出するとすぐに受理される可能性があります。
     *    - その他の Result を返した場合、利用要求の再提出に対する結果を予測できないため、直前のシーケンスに戻ることを推奨します。
     *  この関数は成功を返しません。
     */
    nn::Result FinishHandlingNetworkRequestResult(const nn::applet::LibraryAppletHandle& libraryAppletHandle) NN_NOEXCEPT;

    /**
     * @brief 利用要求の結果に対するエラーハンドリングを行った際の応答を取得します
     *
     * @param[in]  storageHandle              ストレージハンドル
     *
     * @retresult
     *   @handleresult{
     *     ResultErrorHandlingCompleted, エラーハンドリングが正常に行われました。エラーが解消された可能性があります。
     *   }
     *   - その他の Result の場合、エラーハンドリングに失敗しています。
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - storageHandle が有効な値
     *
     * @details
     *  この関数は OceanFramework を利用するアプレットが Unwound された後に呼び出します。
     *  FinishHandlingNetworkRequestResult() はライブラリアプレットハンドルを閉じるので、その後に呼び出すことはできません。
     *  この関数が
     *    - ResultErrorHandlingCompleted を返した場合、利用要求を再提出するとすぐに受理される可能性があります。
     *    - その他の Result を返した場合、利用要求の再提出に対する結果を予測できないため、直前のシーケンスに戻ることを推奨します。
     *  この関数は成功を返しません。
     *
     * @note
     *   Ocean LA でエラーハンドリングを行うフローは、基本的には以下のようになります。
     *   - nn::nifm::PrepareHandlingNetworkRequestResult() の呼び出しが成功すれば、 nn::applet::LibraryAppletHandle を取得できます。
     *   - nn::applet::GetLibraryAppletId() で AppletId を取得できます。
     *   - AppletId から Winding の要・不要を判定して nn::la::LibraryAppletStartHookUserArg を構成します。また、必要ならば自分自身が呼び出されることを避けてください。
     *   - 構成された LibraryAppletStartHookUserArg を引数として nn::nifm::StartHandlingNetworkRequestResult() を呼び出します。
     *       - Winding される場合、上記 API は返らずに次の LA に処理が移ります。
     *           - Unwound 後には GetResultHandlingNetworkRequestResult() を呼び出してハンドリング結果を取得します。
     *       - Winding されない場合、上記 API が成功すれば nn::nifm::FinishHandlingNetworkRequestResult() を呼び出して結果を取得します。
     */
    nn::Result GetResultHandlingNetworkRequestResult(const nn::applet::StorageHandle& storageHandle) NN_NOEXCEPT;

    //! @}
}
}

