﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/util/nifm_EventHandler.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/util/util_Optional.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class RequestManager;
class ConnectionSelector;
class EthernetInterfaceManager;

class PowerStateCoordinator
{
public:
    enum class State
    {
        FullAwake,
        MinimumAwake,
        Sleep,
        Shutdown,
    };

private:
    RequestManager* m_pRequestManager;
    ConnectionSelector* m_pConnectionSelector;
    EthernetInterfaceManager* m_pEthernetInterfaceManager;

    mutable nn::os::SdkMutex m_Mutex;  // 以下の変数の排他を取る Mutex

    nn::psc::PmModule m_PmModule;
    nn::psc::PmState m_PmState;
    nn::psc::PmFlagSet m_PmFlagSet;

    // PSC からのイベントハンドリングに利用
    class PowerStateEventCallback : public CallbackObject
    {
    private:
        PowerStateCoordinator* m_pPowerStateCoordinator;

    public:
        explicit PowerStateEventCallback(PowerStateCoordinator* pPowerStateCoordinator)
            : m_pPowerStateCoordinator(pPowerStateCoordinator)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            m_pPowerStateCoordinator->PowerStateEventCallback();
        }
    } m_PowerStateEventCallback;
    nn::util::optional<SingleSystemEventHandler> m_PowerStateEventHandler;

    State m_CurrentState;
    State m_GoalState;

    // PSC イベントハンドリングの結果を受けての DispatchLoop 駆動に利用
    nn::os::Event m_DispatchTriggerEvent;
    SingleEventHandler m_DispatchTriggerEventHandler;

public:
    PowerStateCoordinator(
        RequestManager* pRequestManager,
        ConnectionSelector* pConnectionSelector,
        EthernetInterfaceManager* pEthernetInterfaceManager
    ) NN_NOEXCEPT;
    ~PowerStateCoordinator() NN_NOEXCEPT;

    void Dispatch() NN_NOEXCEPT;

    State GetGoalState() const NN_NOEXCEPT;

    void AcknowledgeSleep() NN_NOEXCEPT;
    void AcknowledgeAwake() NN_NOEXCEPT;
    void AcknowledgeShutdown() NN_NOEXCEPT;

    // For Test
    Result PutToSleep() NN_NOEXCEPT;
    Result WakeUp() NN_NOEXCEPT;
    Result Shutdown() NN_NOEXCEPT;

    SingleSystemEventHandler& GetPowerStateEventHandler() NN_NOEXCEPT
    {
        return *m_PowerStateEventHandler;
    }

    EventHandler& GetDispatchTriggerEventHandler() NN_NOEXCEPT
    {
        return m_DispatchTriggerEventHandler;
    }

private:
    void PowerStateEventCallback() NN_NOEXCEPT;
};

}
}
}
