﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.horizon.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_Heap.h>

#include <nn/eth/eth_EthTypes.h>
#include <nn/eth/eth_EthClient.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>

#include <nn/util/util_IntrusiveList.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class EthernetInterfaceManagerBase
{
    NN_DISALLOW_COPY(EthernetInterfaceManagerBase);
    NN_DISALLOW_MOVE(EthernetInterfaceManagerBase);

private:
    static const int EthernetInterfaceCountMax = 1;     // 同時に扱える USB Ether アダプターの最大数

private:
    mutable nn::os::SdkMutex m_Mutex;

    // USB Ethernet アダプターの出現・消失をハンドリングするためのメンバ

    mutable nn::eth::client::InterfaceGroupHandler m_InterfaceGroupHandler;     // TODO: Get 系 API が const になったら mutable をはずす
    class InterfaceGroupHandlerInitializer
    {
    public:
        InterfaceGroupHandlerInitializer(
            nn::eth::client::InterfaceGroupHandler& interfaceGroupHandler,
            nn::os::EventClearMode eventClearMode
        )
        {
            interfaceGroupHandler.Initialize(eventClearMode);
        }
    } m_InterfaceGroupHandlerInitializer;

    nn::os::SystemEvent& m_InterfaceGroupEvent;
    class InterfaceGroupEventCallback : public CallbackObject
    {
    private:
        EthernetInterfaceManagerBase* m_pEthernetInterfaceManagerBase;

    public:
        explicit InterfaceGroupEventCallback(EthernetInterfaceManagerBase* pEthernetInterfaceManagerBase)
            : m_pEthernetInterfaceManagerBase(pEthernetInterfaceManagerBase)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE;
    } m_InterfaceGroupEventCallback;
    SingleSystemEventHandler m_InterfaceGroupEventHandler;

    nn::util::IntrusiveList<EthernetInterface, nn::util::IntrusiveListBaseNodeTraits<EthernetInterface>> m_EthernetInterfaceList;
    UnitHeap<EthernetInterface, EthernetInterfaceCountMax> m_EthernetInterfaceHeap;

    // 起動・ウェイク直後におこなう USB Ethernet アダプターの検出待ちの時間切れをハンドリングするためのメンバ

    bool m_IsStandingBy;
    nn::os::TimerEvent m_StandByTimerEvent;
    class StandByTimerEventCallback : public CallbackObject
    {
    private:
        EthernetInterfaceManagerBase* m_pEthernetInterfaceManagerBase;

    public:
        explicit StandByTimerEventCallback(EthernetInterfaceManagerBase* pEthernetInterfaceManagerBase)
            : m_pEthernetInterfaceManagerBase(pEthernetInterfaceManagerBase)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE;
    } m_StandByTimerEventCallback;
    SingleTimerEventHandler m_StandByTimerEventHandler;

    MultiEventHandler m_PluggingEventHandler;

    bool m_IsEnabled;

public:
    // 現在の環境に接続された有線 NIC の MAC アドレスを列挙します。
    // 実機環境ではターゲットデバイスに接続されたすべての有線 NIC が対象になりますが、
    // 利用の候補とならない NIC の MAC アドレスは無効な値 (00:00:00:00:00:00) が入ります。
    // pOutCount はすべての有線 NIC の数を返し、 inCount の値より大きくなることがあります。
    void GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) const NN_NOEXCEPT;

public:
    EthernetInterfaceManagerBase() NN_NOEXCEPT;

    virtual ~EthernetInterfaceManagerBase() NN_NOEXCEPT;

    // USB Ether アダプターの挿抜イベントハンドラへの参照を取得します
    // これのコールバックを呼ぶことで、派生クラスで実装したコールバックが実行されます
    EventHandler& GetPluggingEventHandler() NN_NOEXCEPT
    {
        return m_PluggingEventHandler;
    }

    virtual void AppendCallback(EthernetInterface& ethernetInterface) NN_NOEXCEPT = 0;
    virtual void RemoveCallback(EthernetInterface& ethernetInterface) NN_NOEXCEPT = 0;

    nn::Result SetEnabled(bool isEnabled) NN_NOEXCEPT;
    bool IsEnabled() const NN_NOEXCEPT;

    // Ethernet インターフェースの挿抜状態を最新に反映させます
    // 新しい Ethernet インターフェースを発見した場合は true を返します
    bool Renew() NN_NOEXCEPT;

    // pMultiWaitHolder が指す多重待ちオブジェクトホルダーを保持するインターフェースを探し、
    // 見つかればそのインターフェースへのポインタを、見つからなければ nullptr を返します。
    // この関数の呼び出し中、および取り出したインターフェースへの操作中には
    // インターフェースの増減が発生するような操作をおこなってはいけません。
    EthernetInterface* Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;

    // スリープ明け直後の、 USB Ethernet アダプターの検出を待っている期間か
    bool IsStandingBy() NN_NOEXCEPT;

    nn::Result PutToSleep() NN_NOEXCEPT;
    nn::Result WakeUp() NN_NOEXCEPT;

private:
    void StandByForEthernetInterface() NN_NOEXCEPT;
    void GiveUpEthernetInterface() NN_NOEXCEPT;

protected:
    // 派生クラスのデストラクタから呼んでください
    // 内部で RemoveCallback を呼びます
    void RemoveAll() NN_NOEXCEPT;
};

}
}
}

