﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   NFP(amiibo)に関する非公開 API で利用する型や定数の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nfp/nfp_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/mii/mii_PrivateCommon.h>

namespace nn {
namespace nfp {

/**
 * @brief   プラットフォームを示す値です。
 */
enum PlatformType
{
    PlatformType_Ctr    = 0x00,   //!< CTR
    PlatformType_Cafe   = 0x01,   //!< Cafe
    PlatformType_Snake  = 0x02,   //!< SNAKE
    PlatformType_Nx     = 0x03    //!< NX
};

/**
 * @brief   登録情報のビットフィールドを示す列挙型です。
 */
enum RegisterInfoFlags
{
    RegisterInfoFlags_CommonArea      = 0x01,   //!< 初期登録済み
    RegisterInfoFlags_ApplicationArea = 0x02    //!< アプリケーション専用領域登録済み
};

/**
 * @brief   NFP タグのフォーマットバージョンの値を示す列挙型です。
 */
enum FormatVersion
{
    FormatVersion_NoftV2 = 2      //!< NFP NOFT (Ver 2)
};

/**
 * @brief   amiibo 設定のモードです。
 */
enum AmiiboSettingsMode
{
    AmiiboSettingsMode_NicknameAndOwnerSettings = 0,  //!< ニックネームとオーナーの設定
    AmiiboSettingsMode_GameDataEraser,                //!< ゲームデータの消去
    AmiiboSettingsMode_Restorer,                      //!< データの復旧
    AmiiboSettingsMode_Formatter,                     //!< フォーマット
};

/**
 * @brief   amiibo 設定で扱う情報の有無をビットフィールドを示す列挙型です。
 */
enum AmiiboSettingsInfoFlags
{
    AmiiboSettingsInfoFlags_None          = 0,            //!< 情報なし
    AmiiboSettingsInfoFlags_DeviceHandle  = 1 << 0,       //!< NFC デバイスのハンドル
    AmiiboSettingsInfoFlags_TagInfo       = 1 << 1,       //!< タグ情報
    AmiiboSettingsInfoFlags_RegisterInfo  = 1 << 2,       //!< 登録情報
};

/**
 * @brief   NFP 共用領域の登録情報を示す構造体です。
 */
struct RegisterInfoPrivate
{
    nn::mii::StoreData miiData;                            //!< Mii データ
    Date               registerDate;                       //!< 登録日です。
    char               nickname[NicknameLengthMax * 4 + 1];//!< ニックネーム (+ 終端文字)。UTF-8 で格納してください。
    nn::Bit8           fontRegion;                         //!< ニックネームを表示する際のフォントのリージョン情報です。@ref FontRegion の値を格納してください。
    char               _reserved[142];
};

/**
 * @brief   NFP 共用領域の管理情報を示す構造体です。
 */
struct AdminInfo
{
    nn::ncm::ApplicationId applicationId; //!< 専用領域を作成したアプリケーションの ID です。
    nn::Bit32      accessId;         //!< 専用領域のアクセス ID です。
    uint16_t       moveCounter;      //!< 本体をまたいで書き込みを行った回数です。
    nn::Bit8       registerInfo;     //!< 各種情報の登録状態を表すビットフィールドです。各ビットは @ref RegisterInfoFlags で定義されます。
    uint8_t        formatVersion;    //!< タグのフォーマットバージョンです。@ref FormatVersion の値を格納してください。
    nn::Bit8       platform;         //!< プラットフォームの種類です。@ref PlatformType の値を格納してください。
    char           _reserved[47];
};

/**
 * @brief   amiibo 設定開始時に必要となるパラメータを 1 つにまとめた構造体です。
 */
struct StartParamForAmiiboSettings
{
    uint8_t               version;       //!< @ref StartParamForAmiiboSettings のフォーマットバージョンです。
    nn::Bit8              mode;          //!< amiibo 設定のモードです。@ref AmiiboSettingsMode の値を格納してください。
    nn::Bit8              infoFlags;     //!< 各種情報の有無を表すビットフィールドです。各ビットは @ref AmiiboSettingsInfoFlags で定義されます。NFC デバイスのハンドルは必ず「有り」となります。
    nn::Bit8              optionFlags;   //!< amiibo 設定の動作オプションを表すビットフィールドです。各ビットは @ref AmiiboSettingsOptionFlags で定義されます。
    DeviceHandle          deviceHandle;  //!< NFC デバイスのハンドルです。
    TagInfo               tagInfo;       //!< 更新の対象となるタグを示す情報です。
    RegisterInfo          registerInfo;  //!< amiibo 設定開始時点での登録情報です。
    nn::Bit8              _debug[32];    //   デバッグ情報です。
    char                  _reserved2[36];
};

/**
 * @brief   amiibo 設定の結果を 1 つにまとめた構造体です。
 */
struct ReturnValueForAmiiboSettings
{
    nn::Bit8                  infoFlags;         /*!< 各種情報の有無を表すビットフィールドです。各ビットは @ref AmiiboSettingsInfoFlags で定義されます。@n
                                                   タグが更新されていない場合には、@ref AmiiboSettingsInfoFlags_None を設定してください。
                                                 */
    char                      _reserved1[3];
    DeviceHandle              deviceHandle;      /*!< タグを更新した NFC デバイスのハンドルです。タグを更新した場合、必ず設定してください。@n
                                                   必ずしも @ref StartParamForAmiiboSettings で指定した NFC デバイスのハンドルと一致するものではない点に注意してください。
                                                 */
    TagInfo                   tagInfo;           /*!< 更新されたタグを示す情報です。タグを更新した場合に設定してください。@n
                                                   ただし、@ref StartParamForAmiiboSettings にて @a tagInfo に有効な情報が設定されていた場合、設定されていなくても問題ありません。
                                                 */
    RegisterInfo              registerInfo;      /*!< 更新後の登録情報です。@n
                                                   ニックネームとオーナーの設定を行う amiibo 設定にてタグを更新した場合、必ず設定してください。
                                                 */
    char                      _reserved2[36];
};

}  // nfp
}  // nn
