﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      ニュースのバージョン管理モジュールです。
*/
class Version
{
private:
    /*!
        @brief      コンストラクタです。
    */
    Version() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static Version& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(Version, instance);
        return instance;
    }

    /*!
        @brief      システムがサポートしているバージョンかどうかを判定します。

        @param[in]  version バージョン。

        @return     サポートしているバージョンかどうか。
    */
    static bool IsSupported(const NewsVersion& version) NN_NOEXCEPT;

public:
    /*!
        @brief      クリアします。

        @return     処理結果。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      システム更新を要求します。

        @return     処理結果。

        @details
                    現在のニュースシステムバージョンをストレージに書き込みます。@n
                    本関数を呼び出すことで、 @ref Version::IsSystemUpdateRequired は true を返すようになります。
    */
    nn::Result RequireSystemUpdate() NN_NOEXCEPT;

    /*!
        @brief      システム更新を要求しているかどうかを確認します。

        @param[out] outIsRequired   システム更新を要求しているかどうか。

        @return     処理結果。

        @pre
            - outIsRequired != nullptr

        @details
                    システムは、セーブデータに記録されているニュースシステムバージョンと現在のニュースシステムバージョンを比較します。@n
                    両者が一致している場合、システム更新を要求している状態となります。

                    現在のニュースシステムバージョンの方が大きい場合、システム更新済みと判定します。
    */
    nn::Result IsSystemUpdateRequired(bool* outIsRequired) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    int32_t m_Version;
};

}}}}}
