﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      ブロックサイズです。
*/
const int64_t BlockSize = 16 * 1024;

/*!
    @brief      ニュースデータ保存ストレージのサイズです。

    @details
                内訳は以下の通りです。

                - ニュースデータ
                -- news:/data/<user_id>_<news_id>.msgpack
                -- MAX 12MB
                -- 768 blocks: 12MB
                -- ストレージに空きがない場合、古いものから削除する。
                - エントリー情報
                -- 2 blocks: 32KB

                設定値：6400 blocks: 102400KB
*/
const int64_t DataStorageSize = StorageSize;

/*!
    @brief      ニュースデータ保存ストレージのジャーナルサイズです。

    @details
                一番ジャーナルを消費するアクセスは以下の通りです。

                - ニュースデータの書き込み
                -- 作成：2 (ディレクトリ、ファイル)
                -- 書込：768

                合計：770 blocks: 12320KB

                設定値：800 blocks: 12800KB (reserved 30 blocks: 480KB)
*/
const int64_t DataStorageJournalSize = 800 * BlockSize;

/*!
    @brief      システム情報ストレージのサイズです。

    @details
                内訳は以下の通りです。

                - ニュースデータベース
                -- news-sys:/news.db
                -- MAX 512KB
                -- 32 blocks: 512KB
                - タスクデータ
                -- news-sys:/task.bin
                -- 96KB
                -- 6 blocks: 96KB
                - 受信履歴
                -- news-sys:/history.bin
                -- MAX 512KB
                -- 32 blocks: 512KB
                - One2One 通知リスト
                -- news-sys:/one2one.bin
                -- 8KB
                -- 1 blocks: 16KB
                - インクリメンタル ID
                -- news-sys:/id.bin
                -- 8B
                -- 1 block: 16KB
                - パスフレーズ
                -- news-sys:/passphrase.bin
                -- 128B
                -- 1 block: 16KB
                - システム更新要求バージョン
                -- news-sys:/sup.version
                -- 4B
                -- 1 block: 16KB
                - エントリー情報
                -- 2 blocks: 32KB

                合計：76 blocks: 1216KB

                設定値：256 blocks: 4096KB (reserved 180 blocks: 2880KB)
*/
const int64_t SystemStorageSize = 256 * BlockSize;

/*!
    @brief      システム情報ストレージのジャーナルサイズです。

    @details
                一番ジャーナルを消費するアクセスは以下の通りです。

                - 受信履歴の書き込み
                -- 作成：1
                -- 書込：32

                合計：33 blocks: 528KB

                設定値：48 blocks: 768KB (reserved 15 blocks: 240KB)
*/
const int64_t SystemStorageJournalSize = 48 * BlockSize;

/*!
    @brief      ダウンロードストレージのサイズです。

    @details
                内訳は以下の通りです。

                - ニュースデータ
                -- news-dl:/data.msgpack
                -- MAX 12MB
                -- 768 blocks: 12MB
                - ニュースデータメタデータ
                -- news-dl:/data.msgpack.meta
                -- 344B
                -- 1 block: 16KB
                - ニュースリスト
                -- news-dl:/list.msgpack
                -- MAX 1MB
                -- 64 blocks: 1MB
                - ニュースリストメタデータ
                -- news-dl:/list.msgpack.meta
                -- 344B
                -- 1 block: 16KB
                - エントリー情報
                -- 2 blocks: 32KB

                合計：836 blocks: 13376KB

                設定値：900 blocks: 14400KB (reserved 64 blocks: 1024KB)
*/
const int64_t DownloadStorageSize = 900 * BlockSize;

/*!
    @brief      ダウンロードストレージのジャーナルサイズです。

    @details
                一番ジャーナルを消費するアクセスは以下の通りです。

                - ニュースデータのダウンロード
                -- 作成：2 （データ本体、および、メタデータ）
                -- 書込：769

                合計：771 blocks: 12336KB

                設定値：900 blocks: 14400KB (reserved 129 blocks: 2064KB)

                TODO: ニュースデータを 2MB 受信毎にコミットさせることで、ジャーナルサイズを小さくする。
                      現状はリスト受信後にコミットしていない。
*/
const int64_t DownloadStorageJournalSize = 900 * BlockSize;

}}}}}

/*!
    @brief      システム情報ストレージを常にマウントしておくかどうかのフラグです。
*/
#define NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS 1 // NOLINT(preprocessor/const)
