﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ネットワーク帯域測定 API
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/netdiag/netdiag_BandWidthTypes.h>

namespace nn { namespace netdiag {


    /**
     * @name    ネットワーク帯域測定
     * @{
     */

    /**
     * @brief   アップロード帯域測定を行います。
     *
     * @param[out]   pBandWidth  測定した結果を格納します。単位は bytes/sec です。
     * @param[in]    timeOut     タイムアウトを指定します。単位は ミリ秒 です。
     *
     * @retresult
     *      @handleresult{ResultFailedNetwork, ネットワーク接続に失敗して測定できませんでした。}
     *      @handleresult{ResultInterrupted, 計測の途中でキャンセルされました。}
     * @endretresult
     *
     * @pre
     *  - nifm ライブラリが初期化済
     *  - socket ライブラリが初期化済
     *  - libcurl ライブラリが初期化済
     *
     * @details
     *    関数内部で計測を開始してから、timeOut で指定した時間が過ぎるまでに終了しない場合
     *    タイムアウトとなり、途中で処理を打ち切ります。
     *    その場合でもその時点までの転送に基づいて pBandWidth に結果が格納され、ResultSuccess() を返します。
     *    timeOut には、通常は nn::netdiag::TimeoutForUploadMeasuring の指定を行ってください。
     *    0 を指定するとタイムアウトの設定は行いません。
     */
    Result MeasureBandWidthForUpload( BandWidth* pBandWidth, uint32_t timeOut ) NN_NOEXCEPT;

    /**
     * @brief   アップロード帯域測定を行います。
     *
     * @param[out]   pBandWidth  測定した結果を格納します。単位は bytes/sec です。
     * @param[in]    timeOut     タイムアウトを指定します。単位は ミリ秒 です。
     * @param[out]   pIsTimedout 成功時に、タイムアウトが発生していたかを返します。
     *
     * @retresult
     *      @handleresult{ResultFailedNetwork, ネットワーク接続に失敗して測定できませんでした。}
     *      @handleresult{ResultInterrupted, 計測の途中でキャンセルされました。}
     * @endretresult
     *
     * @pre
     *  - nifm ライブラリが初期化済
     *  - socket ライブラリが初期化済
     *  - libcurl ライブラリが初期化済
     *
     * @details
     *    関数内部で計測を開始してから、timeOut で指定した時間が過ぎるまでに終了しない場合
     *    タイムアウトとなり、途中で処理を打ち切ります。
     *    その場合でもその時点までの転送に基づいて pBandWidth に結果が格納され、ResultSuccess() を返します。
     *    timeOut には、通常は nn::netdiag::TimeoutForUploadMeasuring の指定を行ってください。
     *    0 を指定するとタイムアウトの設定は行いません。
     *
     *    ResultSuccess() 時にタイムアウトが発生していたかどうかは pIsTimedout に格納される値で判定することができます。
     *    pIsTimedout に NULL を指定すると格納しません。
     */
    Result MeasureBandWidthForUpload( BandWidth* pBandWidth, uint32_t timeOut, bool* pIsTimedout ) NN_NOEXCEPT;

    /**
     * @brief   ダウンロード帯域測定を行います。
     *
     * @param[out]   pBandWidth  測定した結果を格納します。単位は bytes/sec です。
     * @param[in]    timeOut     タイムアウトを指定します。単位は ミリ秒 です。
     *
     * @retresult
     *      @handleresult{ResultFailedNetwork, ネットワーク接続に失敗して測定できませんでした。}
     *      @handleresult{ResultInterrupted, 計測の途中でキャンセルされました。}
     * @endretresult
     *
     * @pre
     *  - nifm ライブラリが初期化済
     *  - socket ライブラリが初期化済
     *  - libcurl ライブラリが初期化済
     *
     * @details
     *    関数内部で計測を開始してから、timeOut で指定した時間が過ぎるまでに終了しない場合
     *    タイムアウトとなり、途中で処理を打ち切ります。
     *    その場合でもその時点までの転送に基づいて pBandWidth に結果が格納され、ResultSuccess() を返します。
     *    timeOut には、通常は nn::netdiag::TimeoutForDownloadMeasuring の指定を行ってください。
     *    0 を指定するとタイムアウトの設定は行いません。
     */
     Result MeasureBandWidthForDownload( BandWidth* pBandWidth, uint32_t timeOut ) NN_NOEXCEPT;

    /**
     * @brief   ダウンロード帯域測定を行います。
     *
     * @param[out]   pBandWidth  測定した結果を格納します。単位は bytes/sec です。
     * @param[in]    timeOut     タイムアウトを指定します。単位は ミリ秒 です。
     * @param[out]   pIsTimedout 成功時に、タイムアウトが発生していたかを返します。
     *
     * @retresult
     *      @handleresult{ResultFailedNetwork, ネットワーク接続に失敗して測定できませんでした。}
     *      @handleresult{ResultInterrupted, 計測の途中でキャンセルされました。}
     * @endretresult
     *
     * @pre
     *  - nifm ライブラリが初期化済
     *  - socket ライブラリが初期化済
     *  - libcurl ライブラリが初期化済
     *
     * @details
     *    関数内部で計測を開始してから、timeOut で指定した時間が過ぎるまでに終了しない場合
     *    タイムアウトとなり、途中で処理を打ち切ります。
     *    その場合でもその時点までの転送に基づいて pBandWidth に結果が格納され、ResultSuccess() を返します。
     *    timeOut には、通常は nn::netdiag::TimeoutForDownloadMeasuring の指定を行ってください。
     *    0 を指定するとタイムアウトの設定は行いません。
     *
     *    ResultSuccess() 時にタイムアウトが発生していたかどうかは pIsTimedout に格納される値で判定することができます。
     *    pIsTimedout に NULL を指定すると格納しません。
     */
     Result MeasureBandWidthForDownload( BandWidth* pBandWidth, uint32_t timeOut, bool* pIsTimedout ) NN_NOEXCEPT;

    /**
     * @brief   アップロードやダウンロードの帯域測定の中断指示を出します。
     *
     * @details
     *    nn::netdiag::MeasureBandWidthForUpload() や nn::netdiag::MeasureBandWidthForDownload() の
     *    処理を中断します。
     *    アップロードとダウンロードの、どちらの測定に対してもこの API で中断指示を出します。
     *    この関数でブロックされることはありません。
     *    帯域測定を行っていないときにこの関数を呼んだ場合、何も行いません。
     */
     void InterruptMeasureBandWidth() NN_NOEXCEPT;

    /**
     * @}
     */

}} // nn::netdiag
