﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 *  @file
 *  @brief プライベートデータベースアクセス定義です。
 *  @details Miiの設定・取得をするためのアクセサを定義します。
 */

#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/mii/mii_Common.h>
#include <nn/mii/mii_PrivateCommon.h>

namespace nn { namespace mii {

class RandomDesc; //!< 前方参照

/**
 * @brief データベースクラス
 * @details
 */
class PrivateDatabase
{
    NN_DISALLOW_COPY(PrivateDatabase) NN_NOEXCEPT;
    bool operator ==(const PrivateDatabase& rhs) const NN_NOEXCEPT; ///< 比較禁止
    bool operator !=(const PrivateDatabase& rhs) const NN_NOEXCEPT; ///< 比較禁止
public:
    PrivateDatabase() NN_NOEXCEPT;
    ~PrivateDatabase() NN_NOEXCEPT;

    //! @name 初期化・終了処理
    //! @{

    /**
     * @brief データベースが破損しているかチェックする(オートクリア)
     * @return 破損している場合はtrueを返します。
     * @details データベースの破損をチェックして状態を返します。
     *          チェックのタイミングで破損フラグを自動的にクリアします。
     */
    static bool IsBrokenDatabaseWithClearFlag() NN_NOEXCEPT;

    /**
     * @brief 開始処理
     * @return 成功するとnn::ResultSuccessを返す。
     * @post 成功するデータベースが利用可状態になります。
     * @details データベースの使用を開始します。
     *          データベースを利用するには
     *          この関数を呼び出し、処理を成功させる必要があります。
     *          pResultは、nn::ResultSuccess()時に有効です。
     *
     */
    nn::Result Initialize() NN_NOEXCEPT;

    /**
     * @brief 終了処理
     * @details データベースの使用を終了します。
     * @post データベースが利用不可状態になります。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief 状態確認
     * @details データベースの使用状態を確認します。
     * @return データベースの状態を返します。
     * @retval true  利用可
     * @retval false 利用不可
     */
    bool IsAvailable() const NN_NOEXCEPT;

    //! @}

    //! @name 状態確認
    //! @{

    /**
     * @brief 状態更新が発生したかを判定します。
     * @param[in] sourceFlags @ref SourceFlag
     * @return 更新があればtrueを返します。
     * @details Miiに更新・追加・削除が発生したかを判定します。
     *          @n システムが保持するMiiが更新された場合にtrueを返します。
     *          更新は、アプリ・アプレットの遷移が発生したときに起こります。
     *          @n アプリ・アプレットから戻ってきたときにこの関数は、
     *          trueを返す場合があります。
     *          @n このメソッドはワンショットです。
     *          @n 呼び出すと状態がクリアされることに注意してください。
     */
    bool IsUpdated(int sourceFlags) NN_NOEXCEPT;

    //! @}

    //! @name 取得
    //! @{

    /**
     * @brief データベースの容量が満杯かを判定します。
     * @return 満杯ならtrueを返します。
     * @details GetCount(SourceFlag_Database) == DatabaseMiiCountの判定は行わず、こちらのメソッドを使ってください。
     *          GetCount(SourceFlag_Database) != DatabaseMiiCountの場合でも満杯の場合があります。
     */
    bool IsFullDatabase() const NN_NOEXCEPT;

    /**
     * @brief 指定条件の有効なMii件数を取得します。
     * @param[in] sourceFlags @ref SourceFlag
     * @return 有効なMiiの数を返します。
     */
    int GetCount(int sourceFlags) const NN_NOEXCEPT;

    /**
     * @brief 指定した条件のMiiを取得します。
     * @param[out] pOutCount   pElementsに格納した数
     * @param[out] pElements   格納先配列
     * @param[in]  count       pElementsの要素数
     * @param[in]  sourceFlags @ref SourceFlag
     * @retresult
     *     @handleresult{nn::mii::ResultOverflow,countよりもMiiが多い}
     * @endretresult
     * @details 指定した条件のMiiを取得元を含めて取得します。
     *          sourceFlags に 複数の SourceFlag を指定した場合は、次の順番で詰められて配置されます。
     *          @li SourceFlag_Database
     *          @li SourceFlag_Default
     */
    nn::Result Get(int* pOutCount,StoreDataElement* pElements,int count,int sourceFlags) NN_NOEXCEPT;

    /**
     * @brief 指定した条件のMiiを取得します。
     * @param[out] pOutCount   pElementsに格納した数
     * @param[out] pElements   格納先配列
     * @param[in]  count       pElementsの要素数
     * @param[in]  sourceFlags @ref SourceFlag
     * @retresult
     *     @handleresult{nn::mii::ResultOverflow,countよりもMiiが多い}
     * @endretresult
     * @details 指定した条件のMiiを取得元を含めず取得します。
     *          sourceFlags に 複数の SourceFlag を指定した場合は、次の順番で詰められて配置されます。
     *          @li SourceFlag_Database
     *          @li SourceFlag_Default
     */
    nn::Result Get(int* pOutCount,StoreData* pElements,int count,int sourceFlags) NN_NOEXCEPT;

    /**
     * @brief 指定したStoreDataの最新を取得します。
     * @param[in,out] pStoreData  更新対象のStoreData
     * @param[in]     sourceFlags @ref SourceFlag
     * @retresult
     *     @handleresult{nn::ResultSuccess,最新に更新した}
     *     @handleresult{nn::mii::ResultNotUpdated,更新はありません}
     *     @handleresult{nn::mii::ResultNotFound,見つかりませんでした}
     *     @handleresult{nn::mii::ResultBroken}
     * @endretresult
     * @details 指定した @ref SourceFlag でMiiが更新されているかを取得します。
     */
    nn::Result UpdateLatest(StoreData* pStoreData,int sourceFlags) const NN_NOEXCEPT;

    /**
     * @brief 指定したCreateIdを持つMiiのデータベース上のindexを取得します。
     * @param[in] id                MiiのCreateId
     * @param[in] enabledSpecialMii スペシャルMiiが有効か
     * @return 見つかればそのindexを返します。見つからない場合は-1を返します。
     * @details enabledSpecialMii=falseの場合は、スペシャルMiiを除いたindex番号を返します。
     *          nn::mii::Source_Database内のindexです。
     */
    int FindIndex(const nn::mii::CreateId& id,bool enabledSpecialMii) const NN_NOEXCEPT;

    //! @}

    //! @name データベース操作
    //! @{

    /**
     * @brief 指定したMiiのデータベース位置を移動する。
     * @param[in] newIndex 移動先index(データベース上の位置を指定します。)
     * @param[in] id       移動するMiiのid
     * @retresult
     *     @handleresult{nn::mii::ResultNotFound,idが見つからない}
     *     @handleresult{nn::mii::ResultNotUpdated,移動する必要が無い}
     * @endretresult
     * @details src1とsrc2の配置を入れ替えます。
     */
    nn::Result Move(int newIndex,const CreateId& id) NN_NOEXCEPT;

    /**
     * @brief データベースにMiiを追加もしくは更新します。
     * @param[in] storeData 追加するMiiのStoreData
     * @retresult
     *     @handleresult{nn::mii::ResultFull,満杯}
     * @endretresult
     * @details データベースの末尾にMiiを追加します。
     */
    nn::Result AddOrReplace(const StoreData& storeData) NN_NOEXCEPT;

    /**
     * @brief 指定したCreateIdのMiiを削除する
     * @param[in]  id 削除するMiiのCreateId
     * @retresult
     *     @handleresult{nn::mii::ResultNotFound}
     * @endretresult
     */
    nn::Result Delete(const CreateId& id) NN_NOEXCEPT;

    //! @}

    //! @name デバッグ用
    //! @{

    /**
     * @brief データベースファイルを破壊する
     * @details ファイルシステム上のデータベースを破壊します。
     *          @n メモリ上のデータベースは初期化されます。
     *          @n Mii Testing Mode を Enabled に設定してから呼び出す必要があります。
     */
    nn::Result DestroyFile() NN_NOEXCEPT;

    /**
     * @brief データベースファイルを削除します
     * @details ファイルシステム上のファイルを削除します。
     *          @n Mii Testing Mode を Enabled に設定してから呼び出す必要があります。
     */
    nn::Result DeleteFile() NN_NOEXCEPT;

    /**
     * @brief データベース初期化
     * @details データベースを空に変更します。
     *          @n Mii Testing Mode を Enabled に設定してから呼び出す必要があります。
     */
    void Format() NN_NOEXCEPT;

    /**
     * @brief データベースをインポートします。
     * @param[in] pBuffer インポートするデータベースのポインタ
     * @param[in] size    pBufferのサイズ(nn::mii::ImportFileSize)
     * @details 指定したデータベースファイルをインポートします。
     *          @n Mii Testing Mode を Enabled に設定してから呼び出す必要があります。
     */
    nn::Result Import(const void* pBuffer,size_t size) NN_NOEXCEPT;

    /**
     * @brief データベースをエクスポートします。
     * @param[in] pBuffer データベースエクスポート先ポインタ
     * @param[in] size    pBufferのサイズ(nn::mii::ImportFileSize)
     * @details 指定したデータベースをファイルにエクスポートします。
     *          @n Mii Testing Mode を Enabled に設定してから呼び出す必要があります。
     */
    void Export(void* pBuffer,size_t size) const NN_NOEXCEPT;

    //! @}

private:
    mutable nn::util::TypedStorage<void*, 64, 8> m_Storage;
};

}
}
