﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ネット接続アプレットの起動時引数の受け取りをおこなうクラスの非公開ヘッダ
 */

#pragma once

#include <nn/la/la_NifmToNetConnectArguments.h>
#include <nn/ae/ae.h>


namespace nn { namespace la {

/**
 * @brief   NIFM から渡される、ネット接続アプレット起動時引数の受け取りをおこなうクラスです。
 *          これを利用するアプレットは、 libnn_applet.a, libnn_la_callee.a, libnn_ae.a をリンクする必要があります。
 */
class NifmToNetConnectArgumentsReader
{
private:
    bool m_IsInitialized;

    NN_DETAIL_LA_CURRENT_NIFMTONETCONNECT_ARGUMENTS_DATA_TYPE m_NifmToNetConnectArgumentsData;

public:
    /**
     * @brief   コンストラクタです。
     *
     * @details
     */
    NifmToNetConnectArgumentsReader() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     *
     * @details
     */
    ~NifmToNetConnectArgumentsReader() NN_NOEXCEPT;


    /**
     * @brief   入力チャンネルから共通引数の pop を試みます。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *  取得に失敗した場合でも、入力チャンネルの状態に副作用を与えることがあります。
     *  内部で ReadFromStorage も呼び出します。
     */
    bool TryPopFromInChannel() NN_NOEXCEPT;

    /**
     * @brief   入力ストレージから実際にデータを読みます。
     *
     * @param[in] storageHandle 共通データを読み出すストレージのハンドルです。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *  Storage の Pop を外部でおこないたい場合にはこちらを使用します。
     */
    bool ReadFromStorage(nn::ae::StorageHandle storageHandle) NN_NOEXCEPT;

    /**
     * @brief   ネット接続アプレットに渡された、直前の接続試行の結果を表す Result を取得します。
     *
     * @return  ネット接続アプレットに渡された 、直前の接続試行の結果を表す Result が返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     *  得られる Result は、直前の接続試行の失敗結果を表し、ネット接続アプレットの起動理由をユーザに伝えるために利用できます。
     *
     */
    Result GetNetworkConnectionResult() NN_NOEXCEPT;

    /**
    * @brief   ネット接続アプレットに渡された、直前の接続試行時に利用された接続設定 ID を取得します。
    *
    * @return  ネット接続アプレットに渡された、直前の接続試行時に利用された接続設定 ID が返ります。
    *
    * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
    *
    * @details
    *  得られる接続設定 ID は、直前の失敗時に利用された接続設定を表します。
    *  利用する接続設定がない失敗の場合など、無効な ID (== nn::util::InvalidUuid) が返る場合もあります。
    */
    nn::util::Uuid GetNetworkProfileId() NN_NOEXCEPT;
};

}} // namespace nn::la

