﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/ddsf/ddsf_ISession.h>
#include <nn/i2c/driver/i2c_Bus.h>

namespace nn { namespace i2c { namespace driver {

class I2cDeviceProperty;

namespace detail {

class I2cSessionContext : public nn::ddsf::ISession
{
    NN_DDSF_CAST_SAFE_DECL;

    NN_DISALLOW_COPY(I2cSessionContext);
    NN_DISALLOW_MOVE(I2cSessionContext);

public:
    I2cSessionContext(int maxRetryCount, nn::TimeSpan retryInterval) NN_NOEXCEPT;
    virtual ~I2cSessionContext() NN_NOEXCEPT;

    nn::Result Open(I2cDeviceProperty* pDevice, nn::ddsf::AccessMode accessMode) NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;

    nn::Result Send(const void* pData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT;
    nn::Result Receive(void* pOutData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT;
    nn::Result ExecuteCommandList(void* pOutReceiveBuffer, size_t receiveBufferSize, const void* pCommandList, size_t commandListLength) NN_NOEXCEPT;
    nn::Result SetRetryPolicy(int maxRetryCount, int retryIntervalMicroSeconds) NN_NOEXCEPT;

private:
    enum class Command
    {
        Send,
        Receive,
    };

private:
    nn::Result SendHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT;
    nn::Result ReceiveHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT;
    nn::Result ExtensionHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT;

    nn::Result ExecuteTransactionWithRetry(void* pOutData, Command command, const void* pInData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT;

private:
    int m_MaxRetryCount;
    nn::TimeSpan m_RetryInterval;
};

struct NN_ALIGNAS(I2cSessionContextAlign) I2cSessionContextPadded
{
    I2cSessionContext impl;

    // 64bit ビルド時にサイズがかわるため、パディングを入れる。
    char _padding[I2cSessionContextSize - sizeof(I2cSessionContext)];
};

inline I2cSessionContext& GetI2cSessionContext(I2cSession& session) NN_NOEXCEPT
{
    return nn::util::Get<I2cSessionContextPadded>(session._impl).impl;
}

inline I2cSessionContext& GetI2cSessionContextWithOpenCheck(I2cSession& session) NN_NOEXCEPT
{
    auto& impl = GetI2cSessionContext(session);
    NN_SDK_ASSERT(impl.IsOpen());
    return impl;
}

inline const I2cSessionContext& GetI2cSessionContext(const I2cSession& session) NN_NOEXCEPT
{
    return nn::util::Get<I2cSessionContextPadded>(session._impl).impl;
}

inline const I2cSessionContext& GetI2cSessionContextWithOpenCheck(const I2cSession& session) NN_NOEXCEPT
{
    const auto& impl = GetI2cSessionContext(session);
    NN_SDK_ASSERT(impl.IsOpen());
    return impl;
}

//---------------------------------------------------------------------------
// static_assert で TypedStorage のサイズとアラインを確認
//---------------------------------------------------------------------------

NN_STATIC_ASSERT(sizeof(I2cSessionContextPadded) == I2cSessionContextSize);
NN_STATIC_ASSERT(sizeof(I2cSessionContext) <= I2cSessionContextSize);
NN_STATIC_ASSERT(NN_ALIGNOF(I2cSessionContextPadded) == I2cSessionContextAlign);
NN_STATIC_ASSERT(NN_ALIGNOF(I2cSessionContext) <= I2cSessionContextAlign);

}}}} // nn::i2c::driver::detail
