﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_File.h>


namespace nn { namespace http { namespace detail {
class AbstractLocalStorage;
}}} // ~namespace nn::http::detail

namespace nn { namespace http { namespace json {

/** @brief 固定長の文字列バッファに文字列を格納するための InputStreamType の部分実装です。
    @details
        ImportJsonByRapidJson() の InputStreamType を部分的に実装しています。
        このクラスを継承したクラスで InputStreamType を完全に実装してください。

        このクラスは StringBufferForRapidJson::SetStringBuffer() で与えられるバッファの範囲に切り詰めて文字列を取り扱います。
        ここで切り詰められるのはJSONドキュメント中のペアの name と、文字列値の両方です。
        このバッファの範囲に収まらない文字列は、切り詰められた形で取り扱われます。
 */
class StringBufferForRapidJson
{
    NN_DISALLOW_COPY(StringBufferForRapidJson);

public:
    typedef char Ch;

private:
    Ch* m_Buffer;
    size_t m_BufferSize;
    size_t m_FilledSize;

protected:
    StringBufferForRapidJson() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_BufferSize(0u)
    {
        // In-situ でなければ、StringBuffer は nullptr でも構わない。
    }
    ~StringBufferForRapidJson() NN_NOEXCEPT {} // default

public:
    void SetStringBuffer(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(buffer == nullptr || bufferSize > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
    }

    Ch* PutBegin() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(
            m_Buffer,
            "[nn::http] -----------------------------------------------\n"
            "  ABORT: Streaming string output to nullptr. (internal)\n");
        m_FilledSize = 0u;
        return m_Buffer;
    }
    void Put(Ch c) NN_NOEXCEPT
    {
        if (m_FilledSize >= m_BufferSize)
        {
            NN_SDK_ASSERT(m_FilledSize == m_BufferSize);
            return;
        }
        m_Buffer[m_FilledSize ++] = c;
    }
    void Flush() NN_NOEXCEPT  { NN_ABORT("not implemented"); }
    size_t PutEnd(Ch* ptr) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(ptr == m_Buffer);
        NN_UNUSED(ptr);
        NN_ABORT_UNLESS(m_FilledSize > 0 && m_FilledSize <= m_BufferSize);
        if (m_Buffer[m_FilledSize - 1] != '\0')
        {
            m_Buffer[m_FilledSize - 1] = '\0';
            NN_SDK_LOG(
                "[nn::http] INFO: Input string is truncated to length %u. (input data)\n",
                m_FilledSize - 1);
        }
        return m_FilledSize;
    }
};

/** @brief ストリームからの入力を一時的にメモリにプールして IO 効率を向上した InputStreamType の実装です。
    @details
        ImportJsonByRapidJson() の InputStreamType を実装しています。

        このクラスは CRTP を利用した静的インターフェースクラスです。
        派生クラスで BufferedStreamForRapidJson::FillBufferImpl(void* buffer, size_t bufferSize) 関数を実装してください。
        この関数に指定されたバッファにコピーされたデータは、 ImportJsonByRapidJson() への入力として使用されます。
 */
template <typename Impl>
class BufferedStreamForRapidJson
    : public StringBufferForRapidJson
{
    NN_DISALLOW_COPY(BufferedStreamForRapidJson);

private:
    typedef StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;
    NN_STATIC_ASSERT(sizeof(Ch) == 1);

private:
    Ch* m_Buffer;
    size_t m_BufferSize;
    size_t m_TotalBufferedBytes;
    size_t m_BufferedBytes;
    size_t m_PositionInBuffer;

    bool FillBuffer() NN_NOEXCEPT
    {
        m_PositionInBuffer = 0u;
        m_BufferedBytes = static_cast<Impl*>(this)->FillBufferImpl(m_Buffer, m_BufferSize);
        m_TotalBufferedBytes += m_BufferedBytes;
        return m_BufferedBytes > 0;
    }

protected:
    BufferedStreamForRapidJson() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_BufferSize(0u)
        , m_TotalBufferedBytes(0u)
        , m_BufferedBytes(0u)
        , m_PositionInBuffer(0u)
    {
    }

public:
    void SetInputBuffer(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(buffer && bufferSize > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
    }

    inline Ch Peek() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Buffer != nullptr);
        NN_SDK_ASSERT(m_BufferSize > 0);

        if (m_PositionInBuffer >= m_BufferedBytes)
        {
            NN_SDK_ASSERT(m_PositionInBuffer == m_BufferedBytes);
            if (!FillBuffer())
            {
                return '\0';
            }
        }
        return m_Buffer[m_PositionInBuffer];
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++ m_PositionInBuffer;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return (m_TotalBufferedBytes - m_BufferedBytes) + m_PositionInBuffer;
    }
};

}}} // ~namespace nn::http::json
