﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/util/util_StringView.h>
#include <nn/http/http_Types.h>

#include <curl/curl.h>

namespace nn
{
namespace http
{

class Request;
class ConnectionBroker;

/**
 * @brief   HTTP 応答の実装クラスです

 * @details
 *  このクラスを介して HTTP 応答の解析や受信を行います。
 *  HTTP ライブラリがプロセス化された場合はプロセス側に閉じ込められるコードになります。
 */
class ResponseImpl
{
    NN_DISALLOW_COPY(ResponseImpl);
    NN_DISALLOW_MOVE(ResponseImpl);

    friend class ConnectionBroker;
    friend class Request;

public:
    /*
    enum ConnectionState
    {
        ConnectionState_None,
        ConnectionState_Started,
        ConnectionState_ReceivingBody,
        ConnectionState_Completed
    };*/

    virtual ~ResponseImpl();
    virtual void Dispose();

    int16_t GetStatusCode();
    Result GetStatusCodeAsResult();
    int64_t GetContentLength();
    const char* GetHeaderValue(const char* name);

    virtual void OnHeaderReceived(const nn::util::string_view& name, const nn::util::string_view& value);
    virtual void OnTransfer(curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow);

    ConnectionState GetState() const;
    bool IsStarted() const;
    bool IsConnected() const;
    bool IsCompleted() const;
    bool WaitConnected();
    bool WaitCompleted();
    void SetCancelEvent(os::Event* pCancelEvent)
    {
        m_pCancelEvent = pCancelEvent;
    }

    bool IsCancelRequested() const
    {
        return m_bCancelRequested;
    }

protected:
    ResponseImpl();

    Result AssignHandle(CURL* pCurlEasyHandle, ConnectionBroker& connection);
    virtual bool OnFillBuffer(void* pSource, size_t size) = 0;

    void NotifyConnected();
    virtual void NotifyCompletedWithCode(CURLcode code);

    Result GetLastResult() const;
    bool CheckPendingCancelRequest();

    enum CurlCallbackResult
    {
        CurlCallbackResult_Continue = 0,
        CurlCallbackResult_Cancel   = 1
    };

    static ResponseImpl* GetInstanceFromEasyHandle(CURL* pCurlEasyHandle);

    void AttachResponseToEasyHandle(CURL * pCurlEasyHandle);
    static int CurlXferInfoFunction(void * clientp, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow);
    static size_t CurlHeaderFunction(char *pBuffer, size_t size, size_t nmemb, void* pUserdata);
    static size_t CurlWriteFunction(char *pBuffer, size_t size, size_t nmemb, void *pUserdata);

    ConnectionBroker* m_pConnection;
    CURL* m_pCurlEasyHandle;
    CURLcode m_CurlCode;
    os::ConditionVariable m_cond;
    os::Mutex m_mutex;
    os::Mutex m_mutexForCond;

private:
    os::Event* m_pCancelEvent;
    bool m_bCancelRequested;
    ConnectionState m_State;
};

}
}
