﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_TypesFriendRequest.h>
#include <nn/friends/friends_ResultFriendRequest.h>
#include <nn/friends/friends_AsyncContext.h>

namespace nn { namespace friends {

/*!
    @}
    @name フレンド申請
    @{
*/

/*!
    @brief      フレンド申請を行います。

    @param[out] outAsync            非同期コンテキスト。
    @param[in]  uid                 ユーザーアカウント。
    @param[in]  accountId           相手のネットワークサービスアカウント ID。
    @param[in]  inAppScreenName     相手のアプリ内表示名。
    @param[in]  myInAppScreenName   自分のアプリ内表示名。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請の登録キーを取得します。

    @param[out] outKey  登録キー。
    @param[in]  uid     ユーザーアカウント。

    @return     処理結果。

    @pre
        - outKey != nullptr
        - 指定したユーザーアカウントが Open 状態である。

    @details
                対面フレンド申請を行うための登録キーを取得します。

                本関数で取得した登録キーは、 @ref AddFacedFriendRequest で使用します。
*/
nn::Result GetFacedFriendRequestRegistrationKey(FacedFriendRequestRegistrationKey* outKey,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請を追加します。

    @param[in]  uid                 ユーザーアカウント。
    @param[in]  key                 相手の登録キー。
    @param[in]  nickname            相手のニックネーム。
    @param[in]  image               相手のプロフィール画像。
    @param[in]  imageSize           相手のプロフィール画像のサイズ。
    @param[in]  inAppScreenName     相手のアプリ内表示名。
    @param[in]  myInAppScreenName   自分のアプリ内表示名。

    @return     処理結果。

    @pre
        - image != nullptr
        - 0 < imageSize <= ProfileImageSizeMax
        - 指定したユーザーアカウントが Open 状態である。

    @details
                ニックネームとプロフィール画像は、必ず @ref nn::account::GetNickname と @ref nn::account::LoadProfileImage で取得したものを指定してください。@n
                アプリケーション独自のニックネームやプロフィール画像を指定してはいけません。

                対面フレンド申請は本体ストレージ上に保存され、ネットワーク接続時に自動送信されます。@n
                送信に成功した、または、相手がすでにフレンドだった場合、対面フレンド申請は自動送信後に削除されます。

    @see
        - nn::account::GetNickname
        - nn::account::LoadProfileImage
*/
nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const void* image, size_t imageSize,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT;

/*!
    @}
*/

}}
