﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <cstring>
#include <algorithm>

namespace nn { namespace friends { namespace detail { namespace service { namespace util {

/*!
    @brief      配列のアクセサーモジュールです。
*/
class ArrayAccessor
{
public:
    /*!
        @brief      指定したインデックスの要素を削除します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  index       削除したいインデックス。

        @return     削除後の配列の要素数。
    */
    template <class T>
    static inline int RemoveEntry(T* array, int arraySize, int index) NN_NOEXCEPT
    {
        if (index == arraySize - 1)
        {
            return arraySize - 1;
        }

        std::memmove(&array[index], &array[index + 1], sizeof (T) * (arraySize - index - 1));

        return arraySize - 1;
    }

    /*!
        @brief      指定した条件に一致する要素を 1 つ削除します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  entry       削除したい要素。
        @param[in]  compare     比較関数。

        @return     削除後の配列の要素数。
    */
    template <class T, class U, class Compare>
    static inline int RemoveEntry(T* array, int arraySize, const U& entry, Compare compare) NN_NOEXCEPT
    {
        for (int i = 0; i < arraySize; i++)
        {
            if (compare(array[i], entry))
            {
                return RemoveEntry(array, arraySize, i);
            }
        }

        return arraySize;
    }

    /*!
        @brief      要素を先頭に挿入します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  entry       要素。

        @return     配列の要素数。
    */
    template <class T>
    static inline int InsertToTopEntry(T* array, int arraySize, const T& entry) NN_NOEXCEPT
    {
        if (arraySize > 0)
        {
            std::memmove(&array[1], &array[0], sizeof (T) * arraySize);
        }
        array[0] = entry;

        return arraySize + 1;
    }

    /*!
        @brief      要素を先頭に挿入します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  capacity    配列の最大要素数。
        @param[in]  entry       要素。

        @return     配列の要素数。

        @details
                    現在の配列の要素数が最大要素数と一致している場合、末尾の要素を削除します。
    */
    template <class T>
    static inline int InsertToTopEntry(T* array, int arraySize, int capacity, const T& entry) NN_NOEXCEPT
    {
        if (arraySize == capacity)
        {
            arraySize = RemoveEntry(array, arraySize, arraySize - 1);
        }

        return InsertToTopEntry(array, arraySize, entry);
    }

    /*!
        @brief      要素を末尾に挿入します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  entry       要素。

        @return     配列の要素数。
    */
    template <class T>
    static inline int InsertToEndEntry(T* array, int arraySize, const T& entry) NN_NOEXCEPT
    {
        array[arraySize] = entry;

        return arraySize + 1;
    }

    /*!
        @brief      要素を末尾に挿入します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  capacity    配列の最大要素数。
        @param[in]  entry       要素。

        @return     配列の要素数。

        @details
                    現在の配列の要素数が最大要素数と一致している場合、先頭の要素を削除します。
    */
    template <class T>
    static inline int InsertToEndEntry(T* array, int arraySize, int capacity, const T& entry) NN_NOEXCEPT
    {
        if (arraySize == capacity)
        {
            arraySize = RemoveEntry(array, arraySize, 0);
        }

        return InsertToEndEntry(array, arraySize, entry);
    }

    /*!
        @brief      指定したインデックスの要素を先頭に移動します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  index       移動したいインデックス。
    */
    template <class T>
    static inline void MoveToTopEntry(T* array, int arraySize, int index) NN_NOEXCEPT
    {
        if (index == 0)
        {
            return;
        }

        T entry = array[index];

        std::memmove(&array[1], &array[0], sizeof (T) * index);
        array[0] = entry;
    }

    /*!
        @brief      指定したインデックスの要素を末尾に移動します。

        @param[in]  array       配列。
        @param[in]  arraySize   配列の要素数。
        @param[in]  index       移動したいインデックス。
    */
    template <class T>
    static inline void MoveToEndEntry(T* array, int arraySize, int index) NN_NOEXCEPT
    {
        if (index == arraySize - 1)
        {
            return;
        }

        T entry = array[index];

        std::memmove(&array[index], &array[index + 1], sizeof (T) * (arraySize - index - 1));
        array[arraySize - 1] = entry;
    }
};

}}}}}
