﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/json/friends_JsonInputStream.h>
#include <nn/nn_Result.h>
#include <curl/curl.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

/*!
    @brief      HTTP 通信でサーバーからデータを読み込む入力ストリームです。
*/
class JsonHttpInputStream : public JsonInputStream
{
public:
    typedef JsonInputStream::Ch Ch;

public:
    /*!
        @brief      リクエストヘッダ 1 行のバッファサイズです。
    */
    static const size_t RequestHeaderLineSize = 2048;

    /*!
        @brief      ヘッダの受信コールバックです。

                    処理を中断したい場合、false を返してください。
    */
    typedef bool (*ResponseHeaderCallback)(char* label, char* value, void* param);

    /*!
        @brief      ヘッダの受信完了コールバックです。

                    処理を中断したい場合、false を返してください。
    */
    typedef bool (*ResponseHeaderEndCallback)(int statusCode, void* param);

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonHttpInputStream() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    virtual ~JsonHttpInputStream() NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを開きます。

        @param[in]  url URL。

        @pre
            - url != nullptr

        @return     処理結果。

        @details
                    本関数は、 GET リクエストを発行します。
    */
    nn::Result Open(const char* url) NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを開きます。

        @param[in]  request リクエスト。（GET / PATCH / etc...）
        @param[in]  url     URL。

        @pre
            - request != nullptr
            - url != nullptr

        @return     処理結果。
    */
    nn::Result Open(const char* request, const char* url) NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを閉じます。
    */
    void Close() NN_NOEXCEPT;

    /*!
        @brief      リクエストヘッダを追加します。

        @param[in]  format  フォーマット。
        @param[in]  ...     可変長引数。

        @return     処理結果。
    */
    nn::Result AddRequestHeader(const char* format, ...) NN_NOEXCEPT;

    /*!
        @brief      投稿フィールドを追加します。

        @param[in]  data    データ。
        @param[in]  copy    コピーを作成するかどうか。

        @return     処理結果。
    */
    nn::Result SetPostField(const char* data, bool copy) NN_NOEXCEPT;

    /*!
        @brief      ヘッダの受信コールバックを登録します。

        @param[in]  function    ヘッダの受信コールバック。
        @param[in]  param       コールバックパラメータ。
    */
    void RegisterResponseHeaderCallback(ResponseHeaderCallback function, void* param) NN_NOEXCEPT;

    /*!
        @brief      ヘッダの受信完了コールバックを登録します。

        @param[in]  function    ヘッダの受信完了コールバック。
        @param[in]  param       コールバックパラメータ。

        @details
                    サーバから受信したヘッダをすべて受信した時点で本コールバックは呼び出されます。@n
                    また、ボディデータの受信前に呼び出されれることを保証します。
    */
    void RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback function, void* param) NN_NOEXCEPT;

protected:
    //
    virtual nn::Result FillImpl(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    /*!
        @brief      HTTP データ書き込み関数に渡すパラメータです。
    */
    struct HttpWriteFunctionParam
    {
        void* stream;                //!< 入力ストリームバッファ。
        size_t streamSize;           //!< 入力ストリームバッファのサイズ。
        size_t read;                 //!< 読み込んだサイズ。
        size_t positionInHttpBuffer; //!< HttpWriteFunction で取得できるバッファの読み込み位置。
        bool readNext;               //!< 次のバッファが読み込み可能かどうか。
        bool continuousWrite;        //!< 次のバッファへの書き込みが連続したかどうか。
    };

private:
    //
    CURL* m_Curl;
    CURL* m_Multi;
    //
    curl_slist* m_Headers;
    //
    HttpWriteFunctionParam m_Param;
    //
    ResponseHeaderCallback m_ResponseHeaderCallback;
    void* m_ResponseHeaderParam;
    //
    ResponseHeaderEndCallback m_ResponseHeaderEndCallback;
    void* m_ResponseHeaderEndParam;
    //
    int m_StatusCode;
    //
    bool m_IsOpened;

private:
    //
    static CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT;
    //
    static size_t HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static size_t HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
};

}}}}}
