﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>
#include <nn/friends/detail/service/util/friends_Cancelable.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      タスクのベースクラスです。
*/
class Task : public nn::util::IntrusiveListBaseNode<Task>
{
private:
    NN_DISALLOW_COPY(Task);
    NN_DISALLOW_MOVE(Task);

public:
    /*!
        @brief      タスクの状態を表します。
    */
    enum Status : int8_t
    {
        Status_Initialized            = 0, //!< タスクが作成された状態です。スケジュールされていません。
        Status_Canceled               = 1, //!< タスクがキャンセルされた状態です。
        Status_Waiting                = 2, //!< タスクが実行待機中の状態です。
        Status_Running                = 3, //!< タスクが実行中の状態です。
        Status_Completed              = 4  //!< タスクが完了した状態です。
    };

public:
    /*!
        @brief      コンストラクタです。
    */
    Task() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    virtual ~Task() NN_NOEXCEPT;

    /*!
        @brief      コンテキスト ID を取得します。

        @return     コンテキスト ID。

        @details
                    タスクの生成毎に割り当てられるコンテキスト ID を取得します。@n
                    プロセスが生存している間、一意な値が割り当てられます。
    */
    uint64_t GetContextId() NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントを設定します。

        @param[in]  uid ユーザーアカウント。
    */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントを取得します。

        @return     ユーザーアカウント。
    */
    const nn::account::Uid& GetUid() const NN_NOEXCEPT;

    /*!
        @brief      パラメータを設定します。

        @return     処理結果。

        @details
                    本関数は、タスク実行マクロ（TASK_DO）用のダミーです。
    */
    nn::Result SetParameter() NN_NOEXCEPT;

    /*!
        @brief      タスクを実行します。
    */
    void Run() NN_NOEXCEPT;

    /*!
        @brief      タスクの実行をキャンセルします。
    */
    void Cancel() NN_NOEXCEPT;

    /*!
        @brief      タスクの実行がキャンセルされたかどうかを確認します。

        @return     タスクの実行がキャンセルされたかどうか。
    */
    bool IsCanceled() const NN_NOEXCEPT;

    /*!
        @brief      タスクが完了するまで待機します。
    */
    void Wait() NN_NOEXCEPT;

    /*!
        @brief      タスクが完了するまで待機します。（待機時間指定）

        @param[in]  timeout タイムアウト。

        @return     処理が完了したかどうか。
    */
    bool TimedWait(nn::TimeSpan timeout) NN_NOEXCEPT;

    /*!
        @brief      タスクの状態を設定します。

        @param[in]  status  状態。
    */
    void SetStatus(Status status) NN_NOEXCEPT;

    /*!
        @brief      タスクの状態を取得します。

        @return     状態。
    */
    Status GetStatus() const NN_NOEXCEPT;

    /*!
        @brief      処理結果を取得します。

        @return     処理結果。
    */
    nn::Result GetResult() const NN_NOEXCEPT;

    /*!
        @brief      キャンセル可能オブジェクトを取得します。

        @return     キャンセル可能オブジェクト。
    */
    const detail::service::util::Cancelable& GetCancelable() const NN_NOEXCEPT;

protected:
    /*!
        @brief      タスクの実行関数です。

        @details
                    本クラスの派生クラスは、本メソッドを実装する必要があります。
    */
    virtual nn::Result Main() NN_NOEXCEPT = 0;

private:
    //
    uint64_t m_ContextId;
    //
    nn::os::Event m_Event;
    //
    nn::account::Uid m_Uid;
    //
    detail::service::util::Cancelable m_Cancelable;
    //
    Status m_Status;
    //
    nn::Result m_Result;
};

}}}}}
