﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      パラメータ変換モジュールです。
*/
class ParameterConverter
{
public:
    /*!
        @brief      プレゼンス状態を表す文字列を列挙型に変換します。

        @param[in]  value   文字列値。

        @return     プレゼンス状態。

        @pre
            - value != nullptr
    */
    static PresenceStatus ConvertPresenceStatus(const char* value) NN_NOEXCEPT;

    /*!
        @brief      プレゼンス状態を表す列挙型を文字列に変換します。

        @param[in]  value   プレゼンス状態。

        @return     文字列値。
    */
    static const char* ConvertPresenceStatus(PresenceStatus value) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請手段を表す文字列を列挙型に変換します。

        @param[in]  value   文字列値。

        @return     フレンド申請手段。

        @pre
            - value != nullptr
    */
    static RequestType ConvertRequestType(const char* value) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請手段を表す列挙型を文字列に変換します。

        @param[in]  value   フレンド申請手段。

        @return     文字列値。
    */
    static const char* ConvertRequestType(RequestType value) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請状態を表す列挙型を文字列に変換します。

        @param[in]  value   フレンド申請状態。

        @return     文字列値。
    */
    static const char* ConvertRequestStatus(RequestStatus value) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請状態を表す文字列を列挙型に変換します。

        @param[in]  value   文字列値。

        @return     フレンド申請状態。

        @pre
            - value != nullptr
    */
    static RequestStatus ConvertRequestStatus(const char* value) NN_NOEXCEPT;

    /*!
        @brief      プレゼンスの公開範囲を表す列挙型を文字列に変換します。

        @param[in]  value   プレゼンスの公開範囲。

        @return     文字列値。
    */
    static const char* ConvertPresencePermission(PresencePermission value) NN_NOEXCEPT;

    /*!
        @brief      プレゼンスの公開範囲を表す文字列を列挙型に変換します。

        @param[in]  value   文字列値。

        @return     プレゼンスの公開範囲。

        @pre
            - value != nullptr
    */
    static PresencePermission ConvertPresencePermission(const char* value) NN_NOEXCEPT;

    /*!
        @brief      拡張プロフィールの公開範囲を表す列挙型を文字列に変換します。

        @param[in]  value   拡張プロフィールの公開範囲。

        @return     文字列値。
    */
    static const char* ConvertExtraPermission(ExtraPermission value) NN_NOEXCEPT;

    /*!
        @brief      ブロック理由を表す文字列を列挙型に変換します。

        @param[in]  value   文字列値。

        @return     ブロック理由。

        @pre
            - value != nullptr
    */
    static BlockReason ConvertBlockReason(const char* value) NN_NOEXCEPT;

    /*!
        @brief      ブロック理由を表す列挙型を文字列に変換します。

        @param[in]  value   ブロック理由。

        @return     文字列値。
    */
    static const char* ConvertBlockReason(BlockReason value) NN_NOEXCEPT;

    /*!
        @brief      外部アプリケーションカタログ ID 文字列を変換します。

        @param[out] out     外部アプリケーションカタログ ID。
        @param[in]  value   外部アプリケーションカタログ ID 文字列。

        @return     変換に成功したかどうか。

        @pre
            - out != nullptr
            - value != nullptr
    */
    static bool ConvertExternalApplicationCatalogId(ExternalApplicationCatalogId* out, const char* value) NN_NOEXCEPT;

    /*!
        @brief      プレイログリストを JSON に変換します。

        @param[in]  buffer      バッファ。
        @param[in]  size        バッファサイズ。
        @param[in]  playLogs    プレイログリスト。
        @param[in]  count       プレイログリストの要素数。

        @return     バッファ。

        @pre
            - buffer != nullptr
            - size > 0
    */
    static char* ConvertPlayLog(char* buffer, size_t size, const PlayLog* playLogs, int count) NN_NOEXCEPT;

    /*!
        @brief      2 つのプレイログリストをマージします。

        @param[out] outCount            マージ結果の数。
        @param[out] outMergedPlayLogs   マージされたプレイログリスト。
        @param[in]  basePlayLogs        ベースのプレイログリスト。
        @param[in]  baseCount           ベースのプレイログリストの要素数。
        @param[in]  addPlayLogs         追加するプレイログリスト。
        @param[in]  addCount            追加するプレイログリストの要素数。
        @param[in]  mergedCount         マージされたプレイログリストの要素数。

        @pre
            - outCount != nullptr
            - outMergedPlayLogs != nullptr
            - destCount > 0
    */
    static void MergePlayLog(int* outCount, PlayLog* outMergedPlayLogs,
        const PlayLog* basePlayLogs, int baseCount, const PlayLog* addPlayLogs, int addCount, int mergedCount) NN_NOEXCEPT;
};

}}}}}
