﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      バックグラウンドタスクの管理モジュールです。
*/
class BackgroundTaskManager
{
private:
    NN_DISALLOW_COPY(BackgroundTaskManager);
    NN_DISALLOW_MOVE(BackgroundTaskManager);

public:
    /*!
        @brief      プレゼンスの状態が安定するまでの時間です。

        @details
                    スリープ復帰直後は NPNS によるプレゼンス状態の更新と本プロセスによるプレゼンス状態の更新が順不同で発生する場合があります。@n
                    これを安定させるため、スリープ復帰直後は一定時間ユーザープレゼンスの更新を保留します。
    */
    static const int PresenceStatusStabilizationTime = 10;

    /*!
        @brief      ユーザープレゼンスの最小更新間隔です。

        @details
                    サーバー負荷を軽減するため、ユーザープレゼンスの更新を高頻度で要求されても一定時間保留します。
    */
    static const int PresenceUpdateInterval = 10;

    /*!
        @brief      ユーザープレゼンスの Keep-Alive 間隔です。

        @details
                    ユーザープレゼンスに更新がない場合、一定時間経過するとプレゼンスの状態が自動的にオフライン状態に遷移してしまいます。@n
                    ユーザーアカウントを Open している間、プレゼンスの状態を維持するために Keep-Alive を行います。
    */
    static const int PresenceKeepAliveInterval = 3600;

    /*!
        @brief      ランダムウェイトの範囲（最小値）

        @details
                    負荷分散のためのランダムウェイトの範囲の最小値です。
    */
    static const int RandomWaitMin = 0;

    /*!
        @brief      ランダムウェイトの範囲（最大値）

        @details
                    負荷分散のためのランダムウェイトの範囲の最大値です。
    */
    static const int RandomWaitMax = 15;

    /*!
        @brief      タスク ID です。
    */
    enum TaskId : int32_t
    {
        TaskId_GetListSummary             = 0, //!< リスト情報のサマリー更新を行うタスクです。
        TaskId_SyncUser                   = 1, //!< ユーザー情報を同期するタスクです。
        TaskId_SyncFriendList             = 2, //!< フレンドリストを同期するタスクです。
        TaskId_SyncBlockedUserList        = 3, //!< ブロックリストを同期するタスクです。
        TaskId_SendFacedFriendRequest     = 4, //!< 未送信の対面フレンド申請を送信するタスクです。
        TaskId_DownloadFriendProfileImage = 5, //!< フレンドのプロフィール画像をダウンロードするタスクです。
        TaskId_UpdateUserPresence         = 6, //!< ユーザープレゼンスを更新するタスクです。
        TaskId_GetFriendPresence          = 7, //!< フレンドプレゼンスを取得するタスクです。
        TaskId_GetFriendRequestCount      = 8, //!< フレンド申請件数を取得するタスクです。

        TaskId_Max
    };

    /*!
        @brief      タスクの状態です。
    */
    enum TaskStatus : int32_t
    {
        TaskStatus_Done          = 0, //!< 実行完了。
        TaskStatus_Runnable      = 1, //!< 実行可能状態。
        TaskStatus_Wait          = 2, //!< 待機状態。
        TaskStatus_WaitForOnline = 3  //!< 待機状態。（プレゼンスがオンライン状態になるまで）
    };

    /*!
        @brief      タスク情報です。
    */
    struct TaskInfo
    {
        TaskId taskId;
        nn::account::Uid uid;
        int64_t revision;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    BackgroundTaskManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static BackgroundTaskManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(BackgroundTaskManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      スケジュール処理を行います。

        @details
                    現在の時刻を取得し、次回実行時刻を経過したバックグラウンドタスクが実行可能になります。
    */
    void Schedule() NN_NOEXCEPT;

    /*!
        @brief      実行可能なタスクを 1 つ取得します。

        @param[out] outInfo タスク情報。

        @return     実行可能タスクが存在するかどうか。

        @pre
            - outInfo != nullptr
    */
    bool GetRunnableTask(TaskInfo* outInfo) const NN_NOEXCEPT;

    /*!
        @brief      スケジュール処理を行うべき時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::TimerEvent& GetScheduleEvent() NN_NOEXCEPT;

    /*!
        @brief      タスクの実行完了を通知します。

        @param[in]  info    タスク情報。
        @param[in]  result  処理結果。
    */
    void NotifyDone(const TaskInfo& info, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Open イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserOpened(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Close イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウントの有効性の変化イベントを通知します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  isAvailable 有効かどうか。
    */
    void NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT;

    /*!
        @brief      ユーザープレゼンスの更新を通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserPresenceUpdated(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の受信を通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyFriendRequestReceived(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の承認を通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyFriendRequestAccepted(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンドの削除を通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyFriendDeleted(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請の追加を通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyFacedFriendRequestAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンドリストの同期を要求します。

        @param[in]  uid ユーザーアカウント。

        @pre
            - uid != nn::account::InvalidUid
    */
    void RequestSyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンドのプロフィール画像キャッシュの更新を要求します。

        @param[in]  uid ユーザーアカウント。

        @pre
            - uid != nn::account::InvalidUid
    */
    void RequestUpdateFriendProfileImageCache(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      リスト情報のサマリーの更新を要求します。

        @details
                    本関数を呼び出した時点で Open 状態のユーザーに対して、リスト情報のサマリー更新が行われます。
    */
    void RequestUpdateListSummary() NN_NOEXCEPT;

    /*!
        @brief      スリープを通知します。
    */
    void NotifySleep() NN_NOEXCEPT;

    /*!
        @brief      スリープ復帰を通知します。
    */
    void NotifySleepAwaked() NN_NOEXCEPT;

    /*!
        @brief      NPNS の接続状態の変化を通知します。

        @param[in]  isConnected 接続されているかどうか。
    */
    void NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT;

    /*!
        @brief      フォアグラウンドで通信処理が行われたことを通知します。

        @param[in]  result  通信処理の結果。
    */
    void NotifyForegroundCommunicationDone(nn::Result result) NN_NOEXCEPT;

private:
    //
    struct Task
    {
        TaskStatus status;
        nn::os::Tick lastRunTick;
        nn::os::Tick nextRunnableTick;
        int64_t revision;
    };

    //
    struct Record
    {
        nn::account::Uid uid;
        Task tasks[TaskId_Max];
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::os::TimerEvent m_ScheduleEvent;
    //
    Record m_Records[nn::account::UserCountMax];
    //
    nn::os::Tick m_SleepAwakedTick;
    nn::os::Tick m_ResumeTick;
    //
    bool m_IsConnected;
    bool m_IsSleeping;
    bool m_IsSuspended;
    //
    int32_t m_RetryIntervalTableIndex;

private:
    //
    int FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT;
    //
    void SetRunnable(int index, TaskId taskId) NN_NOEXCEPT;
    void SetRunnableWithoutSignal(int index, TaskId taskId) NN_NOEXCEPT;
    //
    void Wait(int index, TaskId taskId, int wait = 0) NN_NOEXCEPT;
    //
    void SetScheduleTimer() NN_NOEXCEPT;
    //
    bool RescheduleUpdateUserPresenceTask(int index) NN_NOEXCEPT;
    //
    void SuspendTemporarily(int wait) NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;
    //
    static int GetPriority(TaskId taskId) NN_NOEXCEPT;
};

}}}}}
