﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief Error upload API.
*/

#include <nn/eupld/eupld_Types.h>
#include <nn/eupld/eupld_Result.h>
#include <nn/eupld/sfdl/eupld.sfdl.h>

namespace nn    {
namespace eupld {

/**
    @brief      Provides interface for uploading reports
    @details    This class may be used to request a report upload to
                a remote server. There are two types of uploads: i) UploadAll() will upload
                all reports which have not yet been uploaded; ii) UploadSelected() will
                upload specific reports, even if they have already been uploaded.

                Sample code:
                @code
                #include <nn/eupld/eupld_Request.h>

                nn::Result UploadAllSample()
                {
                    nn::Result         result;
                    nn::eupld::Request request;

                    if ((result = request.Initialize()).IsFailure())
                    {
                        return result;
                    }

                    // also see GetResult(), GetEventPointer()
                    result = request.UploadAll();
                    request.Finalize();
                    return result;
                }

                // Report ids may be obtained using nn::erpt::Manager class.
                nn::Result UploadSelectedSample(
                    nn::erpt::ReportId* pReportIdsToUpload,
                    uint32_t            numberOfReportsToUpload)
                {
                    nn::Result         result;
                    nn::eupld::Request request;

                    if ((result = request.Initialize()).IsFailure())
                    {
                        return result;
                    }

                    // also see GetResult(), GetEventPointer()
                    result = request.UploadSelected(pReportIdsToUpload, numberOfReportsToUpload);
                    request.Finalize();
                    return result;
                }
                @endcode
 */
class Request
{
    NN_DISALLOW_COPY(Request);
    NN_DISALLOW_MOVE(Request);

private:
    nn::os::SystemEvent                 m_SystemEvent;
    nn::sf::SharedPointer<sf::IRequest> m_Impl;

public:

    /**
     *  @brief      Upload all previously non-uploaded reports.
     *  @details    This function may be used to initiate upload of all reports which
     *              have not already been uploaded to the server. Once upload has completed,
     *              a system event will be signaled, at which point client may retrieve
     *              result code using GetResult().
     *  @retval     nn::eupld::ResultNotInitialized         Initialize() was not called
     *  @retval     nn::eupld::ResultAlreadyScheduled       Request is already scheduled for upload
     *  @retval     nn::eupld::ResultNoReportsToSend        All reports have already been sent, or there are no reports in the storage.
     *  @retval     nn::eupld::ResultNetworkNotAvailable    Network connection request is not accepted
     *  @pre        - Initialize must be called.
     *              - nifm must be initialized.
     *              - Network connection request must be accepted.
     */
    nn::Result UploadAll()
    NN_NOEXCEPT;

    /**
     *  @brief      Upload specific reports.
     *  @param[out] pReportIds  Array of nn::erpt::ReportId ids for upload. Normally, client would obtain
     *              reportId's for existing reports using nn::erpt::Manager.
     *  @param[out] reportCount Number of reports in reportIds array.
     *  @details    This function may be used to initate upload of specific reports.
     *              Once upload has completed, a system event will be signaled, at which point client
     *              may retrieve result code from upload operation using GetResult().
     *  @retval     nn::eupld::ResultNotInitialized         Initialize() was not called
     *  @retval     nn::eupld::ResultAlreadyScheduled       Request is already scheduled for upload
     *  @retval     nn::eupld::ResultNetworkNotAvailable    Network connection request is not accepted
     *  @pre        - Initialize must be called.
     *              - nifm must be initialized.
     *              - Network connection request must be accepted.
     */
    nn::Result UploadSelected(const nn::erpt::ReportId* pReportIds, uint32_t reportCount)
    NN_NOEXCEPT;

    /**
     *  @brief      Retrieve upload status for each report
     *  @param[in]  reportUploadList  Result information for all transmitted reports.
     *  @details    When upload has failed, client may inspect
     *              upload status for each report that was scheduled for upload to determine
     *              which report has caused the failure. Returned @ref ReportUploadList contains
     *              nn::Result code for each report.
     *  @retval     nn::eupld::ResultNotInitialized     Initialize() was not called
     *  @pre        Initialize must be called.
     */
    nn::Result GetUploadStatus(ReportUploadList& reportUploadList)
    NN_NOEXCEPT;

    /**
     *  @brief      Cancel upload started with UploadAll() or UploadSelected().
     *  @details    Cancels previously started upload.
     *  @retval     nn::eupld::ResultNotInitialized     Initialize() was not called
     *  @retval     nn::eupld::ResultCancelInProgress   Upload is in progress, attempt to cancel is made,
     *                                                  once completed an event will be signaled.
     *  @retval     nn::ResultSuccess                   Upload has succesfully been canceled. In this case
     *                                                  an event will not be signaled.
     *  @pre        Initialize must be called.
     */
    nn::Result CancelUpload()
    NN_NOEXCEPT;

    /**
     *  @brief      Get upload result.
     *  @details    After upload has completed a system event will be signaled. Using GetResult()
     *              client may obtain result of the upload operation.
     *  @retval     nn::eupld::ResultNotInitialized     Initialize() was not called
     *  @pre        Initialize must be called.
     */
    nn::Result GetResult()
    NN_NOEXCEPT;

    /**
     *  @brief      Obtain pointer to the system event.
     *  @details    Client may obtain a pointer to a system event which is signaled
     *              whenever scheduled upload is completed.
     *  @pre        Initialize must be called.
     */
    nn::os::SystemEvent* GetEventPointer()
    NN_NOEXCEPT;

    /**
     *  @brief      Initialize this interface.
     *  @details    Initializes interface to error uploader service. Client must be provisioned
     *              for access to 'eupld:r' port.
     *  @retval     nn::eupld::ResultAlreadyInitialized Initialize has allready been called.
     *  @retval     nn::eupld::ResultOutOfMemory Upload service is out of memory.
     *  @pre        Initialize has not been called.
     */
    nn::Result Initialize()
    NN_NOEXCEPT;

    /**
     *  @brief      Finalize this interface.
     *  @details    Finalizes interface to error uploader service.
     */
    void Finalize()
    NN_NOEXCEPT;

    Request()
    NN_NOEXCEPT;

    ~Request()
    NN_NOEXCEPT;
};

}}
