﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

/**
 *  @file
 *  @brief Error upload control API.
*/

namespace nn    {
namespace eupld {

/**
 *  @brief Maximum length of server URL
 */
const uint32_t MaxUrlLength = 256;

/**
 *  @brief Provides interface for controling upload service
 */
class Control
{
public:

    /**
     *  @brief      Set URL of the report server.
     *  @details    Sets URL of the server. Client needs to be provisioned to access 'eupld:c' port.
     *  @param[out] pUrlString      URL address.
     *  @param[out] urlLength       Size of the URL address.
     *  @retval     nn::eupld::ResultInvalidArgument Passed in URL is longer than MaxUrlLength.
     */
    nn::Result SetUrl(const char* pUrlString, uint32_t urlLength)
    NN_NOEXCEPT;

    /**
     *  @brief      Import certificate for token retrieval.
     *  @details    Imports CA certificate to use for token retrieval.
     *  @param[out] pCrtBuffer      Buffer containing CA certificate.
     *  @param[out] crtBufferLength Length of the cert buffer.
     *  @retval     nn::eupld::ResultOutOfMemory    Report upload service is out of memory.
     */
    nn::Result ImportCrt(const char* pCrtBuffer, uint32_t crtBufferLength)
    NN_NOEXCEPT;

    /**
     *  @brief      Import PKI for token retrieval.
     *  @details    Imports PKI to use for token retrieval.
     *  @param[out] pPkiBuffer      Buffer containing PKI.
     *  @param[out] pkiBufferLength Length of the PKI.
     *  @param[out] pPassword       Buffer containing password.
     *  @param[out] passwordLength  Length of the password.
     *  @retval     nn::eupld::ResultOutOfMemory    Report upload service is out of memory.
     */
    nn::Result ImportPki(const char* pPkiBuffer, uint32_t pkiBufferLength, const char* pPassword, uint32_t passwordLength)
    NN_NOEXCEPT;

    /**
     *  @brief      Enable/disable auto upload of reports.
     *  @details    Function can be used to disable auto upload of reports to remote server.
     *  @param[in]  enable          true/false enables or disabled auto upload.
     *  @param[in]  flagValidTimeSeconds flag's value expires in specified number of seconds.
     */
    nn::Result SetAutoUpload(bool enable, uint32_t flagValidTimeSeconds)
    NN_NOEXCEPT;

    /**
     *  @brief      Get if auto upload of reports is enabled or disabled.
     *  @param[out] pOutIsEnabled           if auto upload is enabled or disabled.
     *  @param[out] pOutValidTimeSeconds    when pOutIsEnabled's value expires
     *  @retval     nn::eupld::ResultOutOfMemory    Report upload service is out of memory.
     */
    nn::Result GetAutoUpload(bool* pOutIsEnabled, uint32_t* pOutValidTimeSeconds)
    NN_NOEXCEPT;
};

}}
