﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ec/system/ec_Async.h>
#include <nn/ec/system/ec_DeviceAccountTypes.h>

namespace nn { namespace ec { namespace system {
    /**
    * @brief    デバイスに保存されているデバイスアカウント情報を取得します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultDeviceAccountNotRegistered    デバイスアカウント情報が保存されていません。
    */
    Result GetDeviceAccountInfo(DeviceAccountInfo* outValue) NN_NOEXCEPT;

    /**
    * @brief    ECI にデバイスアカウントを登録して、得られたデバイスアカウント情報をデバイスに保存します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncResult を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncResult から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxTask          同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired  （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled              （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestRegisterDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    デバイスアカウント情報をデバイスから削除して、ECI のデバイスアカウントを抹消します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncResult を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncResult から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxTask          同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired  （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled              （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestUnregisterDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    ECI からデバイスアカウントの登録情報を取得します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncDeviceRegistrationInfo から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncDeviceRegistrationInfo を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *           戻り値に含まれる DeviceAccountStatus::needsTicketSync にはサーバのチケット同期フラグの状態に関わらず、false が入ります。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncDeviceRegistrationInfo から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultDeviceNotRegistered           デバイスが ECI に登録されていません。
    * @retval   ResultOutOfMaxTask                  同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired          （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    */
    typedef AsyncValue<DeviceRegistrationInfo> AsyncDeviceRegistrationInfo;
    Result RequestDeviceRegistrationInfo(AsyncDeviceRegistrationInfo* outValue) NN_NOEXCEPT;

    /**
    * @brief    デバイスに保存されているデバイスアカウント情報を元に
    *           ECI からデバイスアカウントの状態を取得します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncDeviceAccountStatus から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncDeviceAccountStatus を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *           戻り値に含まれる DeviceAccountStatus::isDeviceTokenExpired にはデバイストークンが実際に失効しているかに関わらず、false が入ります。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncDeviceAccountStatus から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultDeviceAccountNotRegistered    デバイスアカウント情報が保存されていません。
    * @retval   ResultOutOfMaxTask                  同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired          （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    */
    typedef AsyncValue<DeviceAccountStatus> AsyncDeviceAccountStatus;
    Result RequestDeviceAccountStatus(AsyncDeviceAccountStatus* outValue) NN_NOEXCEPT;

    /**
    * @brief    移行元本体からデバイスアカウントを引き継ぎ、ECI からデバイスアカウント情報を取得してデバイスに保存します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncResult を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncResult から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxTask          同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired  （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled              （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestTransferDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    ECI に失効状態のデバイスアカウントの復旧を依頼し、デバイスアカウント情報を取得してデバイスに保存します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncResult を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncResult から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxTask          同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequired  （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
    * @retval   ResultCanceled              （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestSyncRegistration(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    指定したデバイス ID が実行したデバイスのものであるかを判定します。
    */
    bool IsOwnDeviceId(uint64_t deviceId) NN_NOEXCEPT;
}}}
