﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace nim {
    namespace detail {
        class IAsyncValue;
        class IAsyncResult;
        class IAsyncProgressResult;
    }
}}

namespace nn { namespace ec { namespace system {

namespace detail{
    template<typename T>
    class AsyncBase
    {
    public:
        ~AsyncBase() NN_NOEXCEPT;
        void Initialize(sf::SharedPointer<T> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;
        void Wait() NN_NOEXCEPT;
        bool TryWait() NN_NOEXCEPT;
        void Cancel() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        os::SystemEvent& GetEvent() NN_NOEXCEPT;
        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    protected:
        sf::SharedPointer<T> m_Async;

    private:
        util::optional<os::SystemEvent> m_Event;
    };

    class AsyncValueImpl : private detail::AsyncBase<nim::detail::IAsyncValue>
    {
    public:
        ~AsyncValueImpl() NN_NOEXCEPT;
        void Initialize(sf::SharedPointer<nim::detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;
        Result Get(void* buffer, size_t bufferSize) NN_NOEXCEPT;
        void Wait() NN_NOEXCEPT;
        bool TryWait() NN_NOEXCEPT;
        void Cancel() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        os::SystemEvent& GetEvent() NN_NOEXCEPT;
        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    private:
        typedef detail::AsyncBase<nim::detail::IAsyncValue> BaseType;
    };
}

class AsyncResult : private detail::AsyncBase<nim::detail::IAsyncResult>
{
public:

    /**
    * @brief    非同期 Result の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<nim::detail::IAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;

    /**
    * @brief    非同期 Result のデストラクタです。初期化済みのインスタンスには Finalize と同等の処理が行われます。
    */
    ~AsyncResult() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get() NN_NOEXCEPT;

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。
    */
    void Finalize() NN_NOEXCEPT;

    os::SystemEvent& GetEvent() NN_NOEXCEPT;

private:
    typedef detail::AsyncBase<nim::detail::IAsyncResult> BaseType;
};

template<typename T>
class AsyncValue : private detail::AsyncValueImpl
{
public:

    /**
    * @brief    非同期 Value の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<nim::detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Initialize(async, nativeHandle);
    }

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get(T* outValue) NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::Get(outValue, sizeof(T));
    }

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::GetErrorContext(outValue);
    }

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Wait();
    }

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::TryWait();
    }

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Cancel();
    }

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。デストラクタでも同様の処理が行われます。
    */
    void Finalize() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Finalize();
    }

    os::SystemEvent& GetEvent() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::GetEvent();
    }
};

struct AsyncProgress
{
    int64_t done;
    int64_t total;
};

class AsyncProgressResult : private detail::AsyncBase<nim::detail::IAsyncProgressResult>
{
public:

    /**
    * @brief    非同期 Value の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<nim::detail::IAsyncProgressResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;

    /**
    * @brief    デストラクタです。初期化済みのインスタンスには Finalize と同等の処理が行われます。
    */
    ~AsyncProgressResult() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get() NN_NOEXCEPT;

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT;

    /**
    * @brief    対象の進捗を取得します。
    */
    AsyncProgress GetProgress() const NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。デストラクタでも同様の処理が行われます。
    */
    void Finalize() NN_NOEXCEPT;

    os::SystemEvent& GetEvent() NN_NOEXCEPT;

private:
    typedef detail::AsyncBase<nim::detail::IAsyncProgressResult> BaseType;
};

}}}

