﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Eコマース処理を行うAPI向け内部処理関数の定義
 */

#pragma once

#include <nn/ec/ec_ShopTypesForSystem.h>
#include <nn/ec/detail/ec_ShowShopPageArg.h>

#include <string>

namespace nn { namespace ec { namespace detail {

size_t NsUidToString(char* outBuffer, size_t bufferLength, NsUid id) NN_NOEXCEPT;
size_t GetNsUidLengthAsString(NsUid id) NN_NOEXCEPT;
size_t AppIdToString(char* outBuffer, size_t bufferLength, nn::ApplicationId id) NN_NOEXCEPT;
size_t CouponIdToString(char* outBuffer, size_t bufferLength, CouponId id) NN_NOEXCEPT;

// 固定長バッファー用インライン関数

template <size_t N>
inline size_t NsUidToString(char (& outBuffer)[N], NsUid id) NN_NOEXCEPT
{
    return NsUidToString(outBuffer, N, id);
}

template <size_t N>
inline size_t AppIdToString(char (& outBuffer)[N], nn::ApplicationId id) NN_NOEXCEPT
{
    return AppIdToString(outBuffer, N, id);
}

template <size_t N>
inline size_t CouponIdToString(char (& outBuffer)[N], CouponId id) NN_NOEXCEPT
{
    return CouponIdToString(outBuffer, N, id);
}

// GeneratePath 関数用ユーティリティーマクロ

/**
 * @brief GeneratePath 関数でパラメーターの値を指定する際に長さも同時に指定します。
 * @param[in] value パラメーターの値(const char*)
 * @param[in] length パラメーターの長さ(末尾のNULL文字を除く)
 * @details
 * NN_DETAIL_EC_SPECIFY_VALUE_WITH_LENGTH はパラメーターの名前指定に続けて直接指定する必要があります。
 */
#define NN_DETAIL_EC_SPECIFY_PATH_VALUE_WITH_LENGTH(value, length) \
    reinterpret_cast<const char*>(~static_cast<uintptr_t>(0)), static_cast<size_t>((length)), (value)
/**
 * @brief (GeneratePath 関数内部処理用) パラメーターの値と長さを va_list から取り出します。
 * @param[out] outValue パラメーターの値を受け取る const char* 変数へのポインター
 * @param[out] outLength パラメーターの長さを受け取る size_t 変数へのポインター
 * @param[in] v va_list データ
 * @return 長さが指定されていれば true
 */
#define NN_DETAIL_EC_PICK_UP_VALUE_AND_LENGTH(outValue, outLength, v) \
    ((*(outValue) = va_arg(v, const char*)) != reinterpret_cast<const char*>(~static_cast<uintptr_t>(0)) ? false : \
        (*(outLength) = va_arg(v, size_t), *(outValue) = va_arg(v, const char*), true))

/**
 * @brief ECアプレットに渡すパスを生成します。
 * @param[in, out] shopPageArg 生成されたパス文字列を格納するショップ起動パラメーター構造体。
 *   なお、生成されるパス文字列は必ず3071文字(バイト)以下の長さになり、はみ出た分は打ち切られます。
 * @param[in] basePath 基となるパス(ドメインの後ろに付加される、先頭の「/」を除く文字列)
 * @param[in] paramsTail QueryStringの末尾に加える追加パラメーター(先頭に '?' などは不要)。
 *   追加パラメーターが不要である場合は nullptr を指定します。
 * @param[in] ... QueryString に含めるパラメーター名とその値。
 *   「const char* name」と「const char* value」で1セットとし、
 *   必要な数だけこのセットを指定します。
 *   必要な数だけセットしたら、最後の引数に「nullptr」を指定してください。
 * @details
 * - name と value はあらかじめエスケープされている必要があります。
 *   (本関数は簡単のためエスケープ処理を行いません。)
 * - name != nullptr で value == nullptr の場合の動作は未定義です。
 */
void GeneratePath(ShowShopPageArg& shopPageArg, const char* basePath, SourceId sourceId, const char* paramsTail,
    const char* name, const char* value, ...) NN_NOEXCEPT;

/**
 * @brief 文字列がNULL終端されているか判定します。
 * @param[in] str 文字列の先頭ポインター
 * @param[in] length 文字列の長さ
 * @return NULL終端されていれば true
 */
bool IsStringTerminated(char* str, size_t length) NN_NOEXCEPT;
}}}
