﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a RecordIterator class
 * provided by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a RecordIterator class is used to iterate over the
 * nn::dns::parser::MessageSectionConstant::Answer,
 * nn::dns::parser::MessageSectionConstant::Authority, and
 * nn::dns::parser::MessageSectionConstant::Additional sections of the
 * nn::dns::parser::Message and it creates nn::dns::parser::Record
 * objects from the specified section.
 */
class RecordIterator
{
private:
    /**
     * @brief A pointer to the @ref Message.
     */
    const Message* m_pMessage;

    /**
     * @brief A pointer to the section start.
     */
    MemoryBlock m_Section;

    /**
     * @brief A pointer to the current location within the section.
     */
    const uint8_t* m_pCurrent;

    /**
     * @brief The number of records remaining.
     */
    uint8_t m_Count;

    /**
     * @brief The current Record.
     */
    Record m_Record;

public:
    /**
     * @brief Constructor.
     */
    RecordIterator();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The @ref RecordIterator on the right hand side of
     * the expression.
     */
    RecordIterator(const RecordIterator& rhs);

    /**
     * @brief Destructor.
     */
    ~RecordIterator();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The @a RecordIterator on the right hand side of the
     * expression.
     */
    RecordIterator& operator=(const RecordIterator& rhs);

    /**
     * @brief Initializes a RecordIterator.
     *
     * @param[in] message The @ref Message that contains the section.
     *
     * @param[in] section The constant that refers to a specific
     * resource record section.
     *
     * @return This function returns 1 on success, 0 if no records are
     * present, and -1 on error. Errors include:
     *
     * - Message does not have a buffer.
     * - You specify the nn::dns::parser::MessageSectionConstant::Question section.
    */
    int Initialize(const Message& message, MessageSectionConstant section) NN_NOEXCEPT;

    /**
     * @brief This function gets the next Record in the section.
     *
     * This function initializes the 'current' field. If there are
     * additional records then it initializes the record from the cursor
     * and sets the 'message' parameter.
     *
     * @return This function returns 0 if there are no records left then
     * it returns 1 if a single record was cached. If an error is
     * encountered such as exceeding the section boundary then
     * it returns -1.
     */
    int GetNext() NN_NOEXCEPT;

    /**
     * @brief This function gets the current cached @ref Record.
     *
     * @return A reference to the record.
     */
    Record& GetCurrent() NN_NOEXCEPT;
};

}}}; //nn::dns::parser
