﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/os/os_Types.h>
#include <nn/svc/svc_Handle.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/dbg/dbg_Api.h>
#include <nn/osdbg.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs.h>

namespace coredump
{
    struct coredump_compiled_data;
}

namespace nn {
namespace dmnt {

    namespace detail
    {
        const int  MaxSessionCount = 4;
        const int  PoolSessionCount = MaxSessionCount - 2; // Save one interface explicitly for loading and one for debug events
        const char PortName[]      = "dmnt:-";
    }

    Result Initialize() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result BreakDebugProcess( nn::svc::Handle debug ) NN_NOEXCEPT;
    Result TerminateDebugProcess( nn::svc::Handle debug ) NN_NOEXCEPT;
    Result CloseHandle( nn::svc::Handle debug ) NN_NOEXCEPT;

    Result LoadImage( nn::svc::Handle* pOut, const char* pFileName, const char* pArgs ) NN_NOEXCEPT;
    Result AttachByProgramId( nn::svc::Handle* pHandle, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT;
    Result AttachOnLaunch( nn::svc::Handle* pHandle, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT;
    Result GetDebugMonitorProcessId(nn::Bit64* pOut) NN_NOEXCEPT;
    Result GetProcessId( nn::Bit64* pOut, nn::svc::Handle process ) NN_NOEXCEPT;
    Result WaitSynchronization(nn::svc::Handle handle, int64_t ns) NN_NOEXCEPT;
    Result GetDebugEvent(nn::svc::DebugEventInfo* pEvent, nn::svc::Handle process ) NN_NOEXCEPT;
    Result GetProcessModuleInfo(int* pOutCount, nn::dbg::ModuleInfo* pOutModules, int num, nn::os::ProcessId pid) NN_NOEXCEPT;
    Result GetProcessList(int32_t* pNumProcesses, nn::Bit64 pProcessIds[], int32_t arraySize) NN_NOEXCEPT;
    Result GetProcessHandle(nn::svc::Handle* pOut, nn::Bit64 processId) NN_NOEXCEPT;
    Result GetThreadList(int32_t* pNumThreads, nn::Bit64 pThreadIds[], int32_t arraySize, nn::svc::Handle domain) NN_NOEXCEPT;
    Result GetDebugThreadContext(nn::svc::ThreadContext* pContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags) NN_NOEXCEPT;
    Result GetAllDebugThreadInfo( nn::svc::Handle debug, void* pThreadData, nn::Bit32 ElementSize, nn::osdbg::ThreadInfo ThreadInfo[], nn::Bit32 ArraySize ) NN_NOEXCEPT;
    Result ContinueDebugEvent(nn::svc::Handle debug, nn::Bit32 flags, nn::Bit64 threadIds[], nn::Bit32 size) NN_NOEXCEPT;
    Result ReadDebugProcessMemory(uintptr_t buf, nn::svc::Handle debug, uintptr_t addr, size_t size) NN_NOEXCEPT;
    Result WriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t buf, uintptr_t addr, size_t size) NN_NOEXCEPT;
    Result SetDebugThreadContext(nn::svc::Handle debug, nn::Bit64 threadId, const nn::svc::ThreadContext& context, nn::Bit32 controlFlags) NN_NOEXCEPT;
    Result GetDebugThreadParam(nn::Bit64* pOut1, nn::Bit32* pOut2, nn::svc::Handle debug, nn::Bit64 threadId, nn::svc::DebugThreadParam select) NN_NOEXCEPT;
    Result InitializeThreadInfo( nn::osdbg::ThreadInfo* pThreadInfo, nn::svc::Handle debugHandle, const nn::svc::DebugInfoCreateProcess* pDebugInfoCreateProcess, const nn::svc::DebugInfoCreateThread* pDebugInfoCreateThread ) NN_NOEXCEPT;
    Result SetHardwareBreakPoint( nn::svc::HardwareBreakPointRegisterName regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT;
    Result QueryDebugProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle debugHandle, uintptr_t addr) NN_NOEXCEPT;

    Result GetProcessMemoryDetails(int32_t* pNumOfMemoryBlocks, nn::svc::MemoryInfo pBlockInfo[], int32_t arraySize, nn::svc::Handle debugHandle) NN_NOEXCEPT;
    Result GetJitDebugProcessList(int32_t* pNumProcesses, os::ProcessId pProcessIds[], int32_t arraySize) NN_NOEXCEPT;

    //=======================================================================================================================
    // Coredump tasks
    //=======================================================================================================================
    Result CreateCoreDump( char *FileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle ) NN_NOEXCEPT;
    Result InitiateCoreDump( char *FileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle, uint64_t* pDumpHandle ) NN_NOEXCEPT;
    Result ContinueCoreDump( uint64_t DumpHandle, int32_t* pProgress ) NN_NOEXCEPT;
    Result AddTTYToCoreDump( char* pData, uint32_t SizeOfData , uint64_t DumpHandle ) NN_NOEXCEPT;
    Result AddImageToCoreDump( char* pData, uint32_t SizeOfData , uint64_t DumpHandle ) NN_NOEXCEPT;
    Result CloseCoreDump( uint64_t DumpHandle ) NN_NOEXCEPT;

    //=======================================================================================================================
    // Target IO tasks
    //=======================================================================================================================

    // Primitive file operations
    Result TargetIO_FileOpen( const char* pPath, int mode, int32_t CreationDisposition, nn::fs::FileHandle* pHandle ) NN_NOEXCEPT;
    Result TargetIO_FileClose( nn::fs::FileHandle* pHandle ) NN_NOEXCEPT;
    Result TargetIO_FileRead( nn::fs::FileHandle* pHandle, void* pBuffer, int32_t NumberOfBytesToRead, int32_t* pNumberOfBytesRead, int64_t Offset ) NN_NOEXCEPT;
    Result TargetIO_FileWrite( nn::fs::FileHandle* pHandle, void* pBuffer, int32_t NumberOfBytesToWrite, int32_t* pNumberOfBytesWritten, int64_t  Offset ) NN_NOEXCEPT;
    Result TargetIO_FileSetAttributes( const char* pPath, void* pAttribs, int32_t SizeOfAttributes ) NN_NOEXCEPT;

    Result TargetIO_FileGetInformation( const char* pPath, int32_t* pIsDirectory, uint64_t* pSize, uint64_t* pCreateTime, uint64_t* pAccessTime, uint64_t* pModifyTime ) NN_NOEXCEPT;
    Result TargetIO_FileSetTime( const char* pPath, uint64_t CreateTime, uint64_t AccessTime, uint64_t ModifyTime ) NN_NOEXCEPT;

    // Basic file operations
    Result TargetIO_FileSetSize( const char* pPath, int64_t size ) NN_NOEXCEPT;
    Result TargetIO_FileDelete( const char* pPath ) NN_NOEXCEPT;
    Result TargetIO_FileMove( const char* pFromName, const char* pToName ) NN_NOEXCEPT;

    // Directory operations
    Result TargetIO_DirectoryCreate( const char* pPath, nn::fs::OpenMode access ) NN_NOEXCEPT;
    Result TargetIO_DirectoryDelete( const char* pPath ) NN_NOEXCEPT;
    Result TargetIO_DirectoryRename( const char* pFromName, const char* pToName ) NN_NOEXCEPT;

    // Directory search API
    Result  TargetIO_DirectoryGetCount( const char* pPath, int32_t* pNumberOfEntries )  NN_NOEXCEPT;
    Result  TargetIO_DirectoryOpen( const char* pPath, nn::fs::DirectoryHandle* pDirectory )  NN_NOEXCEPT;
    Result  TargetIO_DirectoryGetNext( nn::fs::DirectoryHandle* pDirectory, nn::fs::DirectoryEntry* pEntry )  NN_NOEXCEPT;
    Result  TargetIO_DirectoryClose( nn::fs::DirectoryHandle* pDirectory )  NN_NOEXCEPT;

    // Drive API
    Result TargetIO_GetFreeSpace( int64_t* pFreeBytesAvailable, int64_t* pTotalNumberOfBytes, int64_t* pTotalNumberOfFreeBytes );
    Result TargetIO_GetVolumeInformation( char* pVolumeNameBuffer, int32_t VolumeNameSize, char* pFileSystemNameBuffer, int32_t FileSystemNameSize,
        int32_t* pVolumeSerialNumber, int32_t* pMaximumComponentLength, int32_t* pFileSystemFlags );

} // end of namespace dmnt
} // end of namespace nn
