﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_StaticAssert.h>

#include <nn/nn_Common.h>
#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/codec_OpusEncoderTypes.h>

namespace nn {
namespace codec {
namespace detail {

class OpusPacketInternal
{
public:

    /**
     * @brief       Opus パケットに追加する独自ヘッダの合計サイズです。
     */
    static const size_t HeaderSize = 8;

    /**
     * @brief       Opus パケットに追加する独自ヘッダにおけるサイズ部分へのオフセットです。
     */
    static const size_t OffsetCountToSize = 0;

    /**
     * @brief       Opus パケットに追加する独自ヘッダにおける照合データ部分へのオフセットです。
     */
    static const size_t OffsetCountToFinalRange = 4;

    /**
     * @brief       Opus パケットに追加するヘッダ情報を表す構造体です。
     */
    struct Header
    {
        uint32_t packetSize; //!< Opus パケットサイズ
        uint32_t finalRange; //!< Opus パケットの finalRange
    };

    /**
     * @brief       Opus パケットにヘッダ情報をセットします。
     * @param[out]  pPacket     ヘッダ情報出力先 (Opus パケット)
     * @param[in]   header      ヘッダ情報
     * @pre
     * - pPacket != nullptr
     */
    static void SetHeaderInPacket(uint8_t* pPacket, const struct Header& header) NN_NOEXCEPT;

    /**
     * @brief       Opus パケットからヘッダ情報を取得します。
     * @param[out]  pHeader     ヘッダ情報格納先
     * @param[in]   pPacket     Opus パケット
     * @pre
     * - pPacket != nullptr
     * - pHeader != nullptr
     */
    static void GetHeaderFromPacket(struct Header* pHeader, const uint8_t* pPacket) NN_NOEXCEPT;

    /**
     * @brief       Opus パケットから符号化モードを取得します。
     * @param[out]  pOutCodingMode 符号化モード格納先
     * @param[in]   pPacket        Opus パケット
     * @param[in]   packetSize     pPacket のサイズ
     * @pre
     * - pOutCodingMode != nullptr
     * - pPacket != nullptr
     * - packetSize >= 9
     */
    static OpusResult GetOpusPacketCodingMode(OpusCodingMode* pOutCodingMode, const uint8_t* pPacket, size_t packetSize) NN_NOEXCEPT;
};

NN_STATIC_ASSERT(sizeof(struct OpusPacketInternal::Header) == OpusPacketInternal::HeaderSize);

}}}  // nn::codec::detail
