﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      CDMSC API declaration
*/

#pragma once

namespace nn {
namespace cdmsc {

/**
 * @brief Initialize USB mass storage class library
 *
 * @param[in] pUnitAvailableEvent
 *   An event whill will be signaled when a logical unit is ready to be probed
 *
 * @param[in] alloc
 *   Allocator function pointer
 *
 * @param[in] dealloc
 *   Deallocator function pointer
 *
 * @retresult
 *   @handleresult{ResultSuccess, Initialization succeeded}
 * @endretresult
 *
 * @pre
 *   Initialize() has not already been called, or if it was then Finalize()
 *   was called. pUnitAvailableEvent is not nullptr and it's initialized.
 *
 * @post
 *   Library is ready for operation. The ownership of pUnitAvailableEvent is
 *   transferred to CDMSC and won't be relinquished until Finalize() is called.
 *
 * @details
 *   Upon initialization, the library will internally take ownership of all mass
 *   storage class devices which have been connected to the system. When a
 *   logical unit within the device is enumerated and ready to be used,
 *   pUnitAvailableEvent will be signaled.
 */
Result Initialize(nn::os::EventType *pUnitAvailableEvent,
                  AllocateFunction   alloc,
                  DeallocateFunction dealloc);

/**
 * @brief Finalize USB mass storage class library
 *
 * @retresult
 *   @handleresult{ResultSuccess, Finalization completed}
 * @endretresult
 *
 * @pre
 *   Initialize() has already been called.
 *
 * @post
 *   USB mass storage class library can no longer be used.
 *
 * @details
 *   Upon finalization, the library will internally release ownership of all
 *   mass storage class devices back to the USB host stack.
 */
Result Finalize();

/**
 * @brief Poll for a device which is ready to be used
 *
 * @param[in] pDetachEvent
 *    An event which will be signaled when probed logical unit is detached
 *
 * @param[out] pOutProfile
 *    Profile of probed device
 *
 * @retresult
 *   @handleresult{ResultSuccess, A device is probed}
 *   @handleresult{ResultDeviceNotAvailable, Cannot find any ready device}
 * @endretresult
 *
 * @pre
 *   Initialize() must be called once first. pDetachEvent is not nullptr and
 *   it's initialized. pOutProfile is not nullptr.
 *
 * @post
 *   If a device is probed, the ownership of pDetachEvent is transferred to
 *   CDMSC and won't be relinquished until the probed device is detached.
 *
 * @details
 *   Poll for a logical unit which is ready to be used. This call never blocks.
 *   The profile of probed unit is returned in pOutProfile. pDetachEvent
 *   will be signaled when this probed unit is detached.
 */
Result Probe(nn::os::EventType *pDetachEvent, UnitProfile *pOutProfile);

/**
 * @brief Perform synchronous read operation.
 *
 * @param[out] pOutBuffer
 *   Pointer to user supplied buffer.
 *   Data read from the mass storage device will be written to this buffer.
 *
 * @param[in] handle
 *   Handle of logical unit to be read.
 *
 * @param[in] lba
 *   Starting logical block address to be read.
 *
 * @param[in] blockCount
 *   Number of logical blocks to be read.
 *
 * @retresult
 *   @handleresult{ResultSuccess, Read operation succeeded}
 *   @handleresult{ResultUnitHandleInvalid, Specified unit handle is not valid}
 * @endretresult
 *
 * @pre
 *   pOutBuffer is page aligned and is big enough to hold requested data. lba
 *   and block are within the valid range of the logical unit.
 *
 * @details
 *   Read requested blocks to the buffer.
 */
Result Read(void* pOutBuffer, UnitHandle handle, uint64_t lba, uint32_t block);

/**
 * @brief Perform synchronous write operation.
 *
 * @param[in] pInBuffer
 *   Pointer to user supplied buffer.
 *   Data in this buffer will be written to the logical unit.
 *
 * @param[in] handle
 *   Handle of logical unit to be written.
 *
 * @param[in] lba
 *   Starting logical block address to be written.
 *
 * @param[in] blockCount
 *   Number of logical blocks to be written.
 *
 * @retresult
 *   @handleresult{ResultSuccess, Read operation succeeded}
 *   @handleresult{ResultUnitHandleInvalid, Specified uint handle is not valid}
 * @endretresult
 *
 * @pre
 *   pInBuffer is page aligned. lba and block are within the valid range of the
 *   logical unit.
 *
 * @details
 *   Upon completion of this method, the data could still be in a volatile cache
 *   inside the device, to be written back at a later time, at the discretion of
 *   the device. Flush() must be called to ensure data has been physically
 *   written to the device.
 */
Result Write(const void* pInBuffer, UnitHandle unitHandle, uint64_t lba, uint32_t block);

/**
 * @brief Perform synchronous flush operation.
 *
 * @param[in] handle
 *   Handle of logical unit to be flushed.
 *
 * @param[in] lba
 *   Starting logical block address to be flushed.
 *
 * @param[in] blockCount
 *   Number of logical blocks to be flushed.
 *
 * @retresult
 *   @handleresult{ResultSuccess, Read operation succeeded}
 *   @handleresult{ResultUnitHandleInvalid, Specified unit handle is not valid}
 *   @handleresult{ResultUnsupportedOperation, This unit doesn't support flush operation}
 * @endretresult
 *
 * @pre
 *   lba and block are within the valid range of the logical unit.
 *
 * @details
 *   If the device supports the flush operation, flush the data in device cache
 *   to non-volatile storage.
 */
Result Flush(UnitHandle handle, uint64_t lba, uint32_t block);

}  // cdmsc
}  // nn
